from __future__ import annotations

from dataclasses import dataclass
import os
from dotenv import load_dotenv

load_dotenv()

def _bool(v: str | None, default: bool = False) -> bool:
    if v is None:
        return default
    return v.strip().lower() in {"1", "true", "yes", "y", "on"}

def _int(v: str | None, default: int) -> int:
    try:
        return int(v) if v is not None and v.strip() else default
    except ValueError:
        return default

@dataclass(frozen=True)
class Settings:
    username: str
    password: str

    headless: bool
    slow_mo_ms: int
    timeout_ms: int
    locale: str
    user_agent: str | None

    profile_url: str | None

def get_settings() -> Settings:
    username = os.getenv("IG_USERNAME", "").strip()
    password = os.getenv("IG_PASSWORD", "").strip()

    if not username or not password:
        raise RuntimeError(
            "Missing IG_USERNAME / IG_PASSWORD. Create a .env file from .env.example and set values."
        )

    headless = _bool(os.getenv("IG_HEADLESS"), default=False)
    slow_mo_ms = _int(os.getenv("IG_SLOW_MO_MS"), default=0)
    timeout_ms = _int(os.getenv("IG_TIMEOUT_MS"), default=60000)
    locale = os.getenv("IG_LOCALE", "en-US").strip() or "en-US"
    user_agent = os.getenv("IG_USER_AGENT", "").strip() or None

    profile_url = os.getenv("IG_PROFILE_URL", "").strip() or None

    return Settings(
        username=username,
        password=password,
        headless=headless,
        slow_mo_ms=slow_mo_ms,
        timeout_ms=timeout_ms,
        locale=locale,
        user_agent=user_agent,
        profile_url=profile_url,
    )
