from __future__ import annotations

from dataclasses import dataclass
from playwright.sync_api import sync_playwright, Browser, BrowserContext, Page

from .config import Settings
from .logger import setup_logging

logger = setup_logging()

@dataclass
class Session:
    browser: Browser
    context: BrowserContext
    page: Page

def start_session(settings: Settings) -> Session:
    '''
    Starts a Playwright Chromium session.
    Headful by default (IG_HEADLESS=false) because IG may block headless flows.
    '''
    pw = sync_playwright().start()
    browser = pw.chromium.launch(headless=settings.headless, slow_mo=settings.slow_mo_ms)

    context_kwargs = {
        "locale": settings.locale,
        "viewport": {"width": 1366, "height": 768},
    }
    if settings.user_agent:
        context_kwargs["user_agent"] = settings.user_agent

    context = browser.new_context(**context_kwargs)
    context.set_default_timeout(settings.timeout_ms)

    page = context.new_page()

    # Store playwright object on page for clean shutdown
    setattr(page, "_pw", pw)

    logger.info("Browser session started (headless=%s, slow_mo_ms=%s)", settings.headless, settings.slow_mo_ms)
    return Session(browser=browser, context=context, page=page)

def stop_session(session: Session) -> None:
    try:
        session.context.close()
    finally:
        try:
            session.browser.close()
        finally:
            pw = getattr(session.page, "_pw", None)
            if pw:
                pw.stop()
