from __future__ import annotations

from playwright.sync_api import Page, expect
from ..logger import setup_logging

logger = setup_logging()

IG_BASE = "https://www.instagram.com/"

def dismiss_cookies_if_present(page: Page) -> None:
    selectors = [
        "text=Allow all cookies",
        "text=Allow essential and optional cookies",
        "text=Only allow essential cookies",
        "text=Decline optional cookies",
    ]
    for sel in selectors:
        try:
            btn = page.locator(sel).first
            if btn.is_visible(timeout=1500):
                btn.click(timeout=2000)
                logger.info("Cookie dialog handled via: %s", sel)
                break
        except Exception:
            continue

def login(page: Page, username: str, password: str) -> None:
    '''
    Logs in to Instagram.
    Safe demo only: just login + basic verification.
    '''
    logger.info("Opening Instagram login page...")
    page.goto(IG_BASE, wait_until="domcontentloaded")

    dismiss_cookies_if_present(page)

    logger.info("Waiting for login fields...")
    user_input = page.locator("input[name='username']")
    pass_input = page.locator("input[name='password']")

    user_input.wait_for(state="visible")
    pass_input.wait_for(state="visible")

    user_input.fill(username)
    pass_input.fill(password)

    page.locator("button[type='submit']").click()

    logger.info("Submitted login. Handling post-login dialogs if present...")
    try:
        not_now = page.locator("text=Not Now").first
        if not_now.is_visible(timeout=8000):
            not_now.click()
    except Exception:
        pass

    try:
        not_now2 = page.locator("text=Not Now").first
        if not_now2.is_visible(timeout=4000):
            not_now2.click()
    except Exception:
        pass

    page.wait_for_timeout(1500)
    current_url = page.url
    logger.info("Current URL after login: %s", current_url)

    if "challenge" in current_url or "two_factor" in current_url:
        logger.warning(
            "Instagram requested additional verification (challenge/two_factor). "
            "Complete it manually in the opened browser, then rerun the command."
        )
        return

    try:
        home_icon = page.locator("svg[aria-label='Home'], a[href='/']").first
        expect(home_icon).to_be_visible(timeout=15000)
        logger.info("Login looks successful (home UI detected).")
    except Exception:
        logger.warning(
            "Could not fully confirm home UI. If you see the feed in the browser, it's fine. "
            "IG UI varies by account/region."
        )
