#!/usr/bin/env python
import argparse
import os
import json
import pandas as pd

from app.core.converter import convert_dataframe, load_mapping_config
from app.core.validator import ValidationError

def main():
    p = argparse.ArgumentParser(description="CSV to JSON Converter Pro")
    p.add_argument("--input", "-i", required=True, help="Input CSV path")
    p.add_argument("--out", "-o", required=True, help="Output file path (.json or .ndjson)")
    p.add_argument("--format", "-f", default="json_array", choices=["json_array","ndjson","grouped"], help="Output format")
    p.add_argument("--group-by", default=None, help="Group by column (required for grouped)")
    p.add_argument("--items-key", default="items", help="Items key for grouped output")
    p.add_argument("--delimiter", default=None, help="CSV delimiter")
    p.add_argument("--encoding", default="utf-8", help="CSV encoding")
    p.add_argument("--config", default=None, help="Mapping config JSON path (optional)")
    p.add_argument("--minify", action="store_true", help="Minify JSON (array/grouped)")
    args = p.parse_args()

    if not os.path.exists(args.input):
        raise SystemExit(f"Input file not found: {args.input}")

    df = pd.read_csv(args.input, sep=args.delimiter, encoding=args.encoding)

    mapping = load_mapping_config(args.config) if args.config else {}
    pretty = not args.minify

    try:
        json_text, meta = convert_dataframe(
            df,
            mapping=mapping,
            output_format=args.format,
            group_by=args.group_by,
            items_key=args.items_key,
            pretty=pretty
        )
    except ValidationError as ve:
        raise SystemExit(f"ValidationError: {ve}")
    except Exception as e:
        raise SystemExit(f"Error: {e}")

    os.makedirs(os.path.dirname(args.out) or ".", exist_ok=True)
    with open(args.out, "w", encoding="utf-8") as f:
        f.write(json_text)

    print(f"OK: wrote {args.out}")
    print(json.dumps(meta, indent=2))

if __name__ == "__main__":
    main()
