from __future__ import annotations
from dataclasses import dataclass
from typing import Any, Optional, Dict
from dateutil import parser as dtparser
import re

_BOOL_TRUE = {"true","1","yes","y","t"}
_BOOL_FALSE = {"false","0","no","n","f"}

def _is_null(v: Any) -> bool:
    if v is None:
        return True
    if isinstance(v, float) and v != v:  # NaN
        return True
    if isinstance(v, str) and v.strip() == "":
        return True
    return False

def parse_value(value: Any, forced_type: Optional[str] = None) -> Any:
    """
    Convert a CSV cell to a typed Python value.
    forced_type: one of {"int","float","bool","date","str"} or None for auto-infer.
    """
    if _is_null(value):
        return None

    if isinstance(value, str):
        s = value.strip()
    else:
        s = str(value).strip()

    if forced_type:
        t = forced_type.lower()
        if t == "str":
            return s
        if t == "int":
            return int(float(s))
        if t == "float":
            return float(s)
        if t == "bool":
            sl = s.lower()
            if sl in _BOOL_TRUE:
                return True
            if sl in _BOOL_FALSE:
                return False
            raise ValueError(f"Cannot parse bool from '{s}'")
        if t == "date":
            return dtparser.parse(s).date().isoformat()
        raise ValueError(f"Unknown forced type: {forced_type}")

    # Auto infer
    sl = s.lower()
    if sl in _BOOL_TRUE:
        return True
    if sl in _BOOL_FALSE:
        return False

    # int
    if re.fullmatch(r"[+-]?\d+", s):
        try:
            return int(s)
        except Exception:
            pass

    # float
    if re.fullmatch(r"[+-]?(\d+\.\d+|\d+\.\d*|\d*\.\d+)", s):
        try:
            return float(s)
        except Exception:
            pass

    # date (best-effort)
    # Avoid parsing plain numbers as dates
    if not re.fullmatch(r"[+-]?\d+(\.\d+)?", s):
        try:
            d = dtparser.parse(s, fuzzy=False)
            # keep only date portion for JSON readability
            return d.date().isoformat()
        except Exception:
            pass

    return s

def apply_types(row: Dict[str, Any], types: Optional[Dict[str, str]] = None) -> Dict[str, Any]:
    if not types:
        # auto infer each value
        return {k: parse_value(v) for k, v in row.items()}
    out = {}
    for k, v in row.items():
        out[k] = parse_value(v, types.get(k))
    return out
