import json
import streamlit as st
import pandas as pd

from app.core.converter import read_csv_safely, convert_dataframe, load_mapping_config
from app.core.validator import ValidationError

st.set_page_config(page_title="CSV to JSON Converter", page_icon="🧩", layout="wide")

st.title("CSV to JSON Converter")
st.caption("Upload CSV → Validate/Clean → Map Fields → Export JSON (Array / NDJSON / Grouped)")

with st.sidebar:
    st.header("Options")
    delimiter = st.text_input("Delimiter (optional)", value="", help="Leave empty to auto-detect by pandas.")
    delimiter = delimiter if delimiter.strip() else None

    encoding = st.text_input("Encoding (optional)", value="", help="Try: utf-8, utf-8-sig, latin-1")
    encoding = encoding if encoding.strip() else None

    output_format = st.selectbox("JSON Output Format", ["json_array", "ndjson", "grouped"], index=0)
    pretty = st.checkbox("Pretty JSON (indent)", value=True, help="Applies to JSON Array and Grouped output.")

    st.divider()
    st.subheader("Validation")
    enable_required = st.checkbox("Enforce required fields", value=False)

    st.divider()
    st.subheader("Mapping (Optional)")
    mapping_mode = st.radio("Mapping Source", ["UI Mapping", "Upload mapping.json"], index=0)

uploaded = st.file_uploader("Upload CSV file", type=["csv"])
if not uploaded:
    st.info("Upload a CSV file to begin.")
    st.stop()

try:
    df = read_csv_safely(uploaded.getvalue(), delimiter=delimiter, encoding=encoding)
except Exception as e:
    st.error(f"Failed to read CSV: {e}")
    st.stop()

left, right = st.columns([1.2, 1])

with left:
    st.subheader("CSV Preview")
    st.dataframe(df.head(50), use_container_width=True)

mapping = {}

if mapping_mode == "Upload mapping.json":
    cfg = st.file_uploader("Upload mapping.json", type=["json"], key="mapping_json")
    if cfg:
        try:
            mapping = json.loads(cfg.getvalue().decode("utf-8"))
        except Exception as e:
            st.error(f"Invalid mapping JSON: {e}")
else:
    # UI mapping
    st.subheader("Mapper (UI)")
    st.write("Rename columns or drop columns. Leave rename empty to keep original name.")
    cols = list(df.columns)

    drop_cols = st.multiselect("Drop columns", cols, default=[])

    rename_pairs = {}
    st.write("Rename columns:")
    for c in cols:
        if c in drop_cols:
            continue
        new_name = st.text_input(f"Rename '{c}' to", value=c, key=f"rename_{c}")
        if new_name.strip() and new_name != c:
            rename_pairs[c] = new_name.strip()

    defaults_text = st.text_area("Defaults (JSON object)", value='{}', help='Example: {"Currency":"INR","Source":"Upload"}')
    try:
        defaults_obj = json.loads(defaults_text) if defaults_text.strip() else {}
        if not isinstance(defaults_obj, dict):
            raise ValueError("Defaults must be a JSON object.")
    except Exception as e:
        st.error(f"Defaults JSON error: {e}")
        defaults_obj = {}

    mapping = {
        "rename": rename_pairs,
        "drop": drop_cols,
        "defaults": defaults_obj,
    }

    if enable_required:
        req_cols = st.multiselect("Required fields (after mapping)", [mapping["rename"].get(c,c) for c in cols if c not in drop_cols], default=[])
        mapping["required"] = req_cols

    st.write("Type rules (optional). If empty, auto-infer is used.")
    type_rules_text = st.text_area(
        "Types (JSON object)",
        value="{}",
        help='Example: {"Age":"int","Price":"float","IsActive":"bool","Date":"date","Name":"str"}'
    )
    try:
        types_obj = json.loads(type_rules_text) if type_rules_text.strip() else {}
        if not isinstance(types_obj, dict):
            raise ValueError("Types must be a JSON object.")
        if types_obj:
            mapping["types"] = types_obj
    except Exception as e:
        st.error(f"Types JSON error: {e}")

with right:
    st.subheader("Convert")
    group_by = None
    items_key = "items"
    if output_format == "grouped":
        group_by = st.selectbox("Group by column", list(df.columns), index=0)
        items_key = st.text_input("Items key", value="items")

    convert_btn = st.button("Convert to JSON", type="primary", use_container_width=True)

    if convert_btn:
        try:
            json_text, meta = convert_dataframe(
                df,
                mapping=mapping,
                output_format=output_format,
                group_by=group_by,
                items_key=items_key,
                pretty=pretty
            )
        except ValidationError as ve:
            st.error(str(ve))
            st.stop()
        except Exception as e:
            st.error(f"Conversion failed: {e}")
            st.stop()

        st.success(f"Converted successfully: {meta['rows']} row(s). Invalid rows: {meta['invalid_rows']}")
        if meta["sample_errors"]:
            st.warning("Sample validation issues:")
            for msg in meta["sample_errors"]:
                st.write(f"- {msg}")

        st.subheader("JSON Output")
        if output_format == "ndjson":
            st.code(json_text[:6000] + ("\n...\n" if len(json_text) > 6000 else ""), language="json")
        else:
            try:
                st.json(json.loads(json_text))
            except Exception:
                st.code(json_text, language="json")

        file_name = "output.json" if output_format != "ndjson" else "output.ndjson"
        st.download_button(
            "Download JSON",
            data=json_text.encode("utf-8"),
            file_name=file_name,
            mime="application/json" if output_format != "ndjson" else "application/x-ndjson",
            use_container_width=True
        )

st.divider()
st.caption("Tip: Use mapping.json for repeatable conversions across multiple CSV formats.")
