import axios from 'axios';

const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || 'http://localhost:8080/api';

const api = axios.create({
  baseURL: API_BASE_URL,
  headers: {
    'Content-Type': 'application/json',
  },
});

// Request interceptor to add auth token
api.interceptors.request.use(
  (config) => {
    const token = localStorage.getItem('token');
    if (token) {
      config.headers.Authorization = `Bearer ${token}`;
    }
    return config;
  },
  (error) => {
    return Promise.reject(error);
  }
);

// Response interceptor for error handling
api.interceptors.response.use(
  (response) => response,
  (error) => {
    if (error.response?.status === 401) {
      localStorage.removeItem('token');
      localStorage.removeItem('user');
      window.location.href = '/login';
    }
    return Promise.reject(error);
  }
);

// Auth APIs
export const authAPI = {
  register: (data) => api.post('/auth/register', data),
  login: (data) => api.post('/auth/login', data),
};

// Eligibility APIs
export const eligibilityAPI = {
  checkEligibility: (data) => api.post('/eligibility/check', data),
};

// Loan APIs
export const loanAPI = {
  getMyApplications: () => api.get('/loans/my-applications'),
  getApplicationById: (id) => api.get(`/loans/application/${id}`),
};

// Admin APIs
export const adminAPI = {
  getAnalytics: () => api.get('/admin/analytics'),
  getInterestRates: () => api.get('/admin/interest-rates'),
  createInterestRate: (data) => api.post('/admin/interest-rates', data),
  updateInterestRate: (id, data) => api.put(`/admin/interest-rates/${id}`, data),
  deleteInterestRate: (id) => api.delete(`/admin/interest-rates/${id}`),
  getLoanRules: () => api.get('/admin/loan-rules'),
  createLoanRule: (data) => api.post('/admin/loan-rules', data),
  updateLoanRule: (id, data) => api.put(`/admin/loan-rules/${id}`, data),
};

export default api;
