# Database Schema Documentation

## Database: loanlens_db

### Tables Overview

1. **users** - User authentication and profile
2. **loan_applications** - Loan eligibility applications
3. **interest_rates** - Bank interest rate configurations
4. **loan_rules** - Admin-configurable loan rules

---

## Table: users

Stores user authentication and profile information.

| Column | Type | Constraints | Description |
|--------|------|-------------|-------------|
| id | BIGINT | PRIMARY KEY, AUTO_INCREMENT | Unique user identifier |
| email | VARCHAR(100) | NOT NULL, UNIQUE | User email address |
| password | VARCHAR(255) | NOT NULL | Encrypted password (BCrypt) |
| full_name | VARCHAR(100) | NOT NULL | User's full name |
| phone_number | VARCHAR(15) | NULL | Contact phone number |
| role | ENUM('USER', 'ADMIN') | NOT NULL, DEFAULT 'USER' | User role |
| active | BOOLEAN | NOT NULL, DEFAULT TRUE | Account status |
| created_at | TIMESTAMP | NOT NULL, DEFAULT CURRENT_TIMESTAMP | Account creation time |
| updated_at | TIMESTAMP | NOT NULL, DEFAULT CURRENT_TIMESTAMP ON UPDATE | Last update time |

### Indexes
- PRIMARY KEY (id)
- UNIQUE INDEX (email)

---

## Table: loan_applications

Stores all loan eligibility check applications.

| Column | Type | Constraints | Description |
|--------|------|-------------|-------------|
| id | BIGINT | PRIMARY KEY, AUTO_INCREMENT | Unique application identifier |
| user_id | BIGINT | NOT NULL, FOREIGN KEY | Reference to users table |
| requested_amount | DECIMAL(12,2) | NOT NULL | Loan amount requested by user |
| course_type | ENUM | NOT NULL | Type of course (ENGINEERING, MEDICAL, MBA, etc.) |
| college_tier | ENUM | NOT NULL | College quality tier (TIER_1, TIER_2, TIER_3) |
| co_applicant_income | DECIMAL(12,2) | NOT NULL | Annual income of co-applicant/parent |
| credit_score | INT | NOT NULL | Credit score (300-900) |
| existing_liabilities | DECIMAL(12,2) | NOT NULL | Total existing loan EMIs |
| eligible_amount | DECIMAL(12,2) | NULL | Calculated eligible loan amount |
| eligibility_score | DECIMAL(5,2) | NULL | Eligibility score (0-100) |
| debt_to_income_ratio | DECIMAL(5,2) | NULL | DTI ratio percentage |
| risk_level | ENUM('LOW', 'MEDIUM', 'HIGH') | NULL | Risk assessment |
| created_at | TIMESTAMP | NOT NULL, DEFAULT CURRENT_TIMESTAMP | Application timestamp |

### Course Types
- ENGINEERING
- MEDICAL
- MBA
- LAW
- ARTS
- SCIENCE
- COMMERCE
- OTHERS

### College Tiers
- TIER_1: Premier institutions (IIT, AIIMS, IIM, NLU)
- TIER_2: Good institutions (NIT, State Universities)
- TIER_3: Standard institutions

### Indexes
- PRIMARY KEY (id)
- FOREIGN KEY (user_id) REFERENCES users(id)
- INDEX (user_id, created_at)

---

## Table: interest_rates

Stores bank-wise interest rate information for admin management.

| Column | Type | Constraints | Description |
|--------|------|-------------|-------------|
| id | BIGINT | PRIMARY KEY, AUTO_INCREMENT | Unique rate identifier |
| bank_name | VARCHAR(100) | NOT NULL, UNIQUE | Name of the bank/lender |
| base_rate | DECIMAL(5,2) | NOT NULL | Base interest rate (%) |
| min_rate | DECIMAL(5,2) | NULL | Minimum applicable rate (%) |
| max_rate | DECIMAL(5,2) | NULL | Maximum applicable rate (%) |
| processing_fee_percentage | DECIMAL(5,2) | NULL | Processing fee (% of loan) |
| active | BOOLEAN | NOT NULL, DEFAULT TRUE | Whether rate is active |
| updated_at | TIMESTAMP | NOT NULL, DEFAULT CURRENT_TIMESTAMP ON UPDATE | Last update time |
| special_offer | TEXT | NULL | Special offer details |

### Indexes
- PRIMARY KEY (id)
- UNIQUE INDEX (bank_name)

---

## Table: loan_rules

Stores admin-configurable loan eligibility rules.

| Column | Type | Constraints | Description |
|--------|------|-------------|-------------|
| id | BIGINT | PRIMARY KEY, AUTO_INCREMENT | Unique rule identifier |
| rule_name | VARCHAR(100) | NOT NULL, UNIQUE | Rule name |
| rule_type | VARCHAR(50) | NOT NULL | Type: CREDIT_SCORE, DTI_RATIO, MAX_LOAN |
| min_value | DECIMAL(12,2) | NULL | Minimum threshold value |
| max_value | DECIMAL(12,2) | NULL | Maximum threshold value |
| threshold | DECIMAL(5,2) | NULL | Threshold percentage/value |
| description | TEXT | NULL | Rule description |
| active | BOOLEAN | NOT NULL, DEFAULT TRUE | Whether rule is active |
| updated_at | TIMESTAMP | NOT NULL, DEFAULT CURRENT_TIMESTAMP ON UPDATE | Last update time |

### Rule Types
- CREDIT_SCORE: Credit score validation rules
- DTI_RATIO: Debt-to-income ratio thresholds
- MAX_LOAN: Maximum loan amount limits
- MIN_INCOME: Minimum income requirements

### Indexes
- PRIMARY KEY (id)
- UNIQUE INDEX (rule_name)

---

## Relationships

```
users (1) ----< (many) loan_applications
```

- One user can have multiple loan applications
- Each loan application belongs to one user
- Foreign key constraint with ON DELETE CASCADE

---

## Sample Data

### Default Admin User
```sql
INSERT INTO users (email, password, full_name, phone_number, role, active)
VALUES (
  'admin@loanlens.com',
  '$2a$10$xwVzYiAXVCKxBQ4H6U3lTe9bQrMwKq8v4GNh8kF2PzG7xYzI5jKoK', -- password: admin123
  'Admin User',
  '9999999999',
  'ADMIN',
  true
);
```

### Sample Interest Rates
```sql
INSERT INTO interest_rates (bank_name, base_rate, min_rate, max_rate, processing_fee_percentage, active)
VALUES
  ('State Bank of India', 8.50, 8.00, 10.00, 1.00, true),
  ('HDFC Bank', 8.75, 8.25, 10.50, 1.50, true),
  ('ICICI Bank', 8.60, 8.10, 10.25, 1.25, true),
  ('Axis Bank', 8.80, 8.30, 10.75, 1.75, true),
  ('Punjab National Bank', 8.40, 7.90, 9.90, 0.75, true);
```

### Sample Loan Rules
```sql
INSERT INTO loan_rules (rule_name, rule_type, min_value, max_value, threshold, description, active)
VALUES
  ('Min Credit Score', 'CREDIT_SCORE', 300, 900, 650, 'Minimum credit score required for loan approval', true),
  ('Max DTI Ratio', 'DTI_RATIO', 0, 100, 40, 'Maximum debt-to-income ratio allowed', true),
  ('Max Loan Amount', 'MAX_LOAN', 50000, 20000000, NULL, 'Maximum loan amount per application', true),
  ('Min Annual Income', 'MIN_INCOME', 0, NULL, 200000, 'Minimum annual income of co-applicant', true);
```

---

## Data Constraints & Validations

### loan_applications
- requested_amount: Must be between ₹50,000 and ₹2,00,00,000
- credit_score: Must be between 300 and 900
- co_applicant_income: Must be >= 0
- existing_liabilities: Must be >= 0

### interest_rates
- All rate percentages must be > 0
- min_rate <= base_rate <= max_rate

### Calculated Fields
These are computed by the application and stored:
- eligible_amount: Calculated based on income, credit score, course, and college tier
- eligibility_score: 0-100 score based on multiple factors
- debt_to_income_ratio: (existing_liabilities / annual_income) * 100
- risk_level: Determined by DTI ratio and credit score

---

## Query Examples

### Get User's All Applications
```sql
SELECT * FROM loan_applications
WHERE user_id = ?
ORDER BY created_at DESC;
```

### Get Applications by Risk Level
```sql
SELECT risk_level, COUNT(*) as count
FROM loan_applications
WHERE risk_level IS NOT NULL
GROUP BY risk_level;
```

### Average Eligibility Score
```sql
SELECT AVG(eligibility_score) as avg_score
FROM loan_applications
WHERE eligibility_score IS NOT NULL;
```

### Active Interest Rates
```sql
SELECT * FROM interest_rates
WHERE active = true
ORDER BY base_rate ASC;
```

### Recent Applications with User Details
```sql
SELECT 
  la.*,
  u.full_name,
  u.email
FROM loan_applications la
JOIN users u ON la.user_id = u.id
WHERE la.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
ORDER BY la.created_at DESC;
```

---

## Backup & Maintenance

### Daily Backup
```bash
mysqldump -u root -p loanlens_db > backup_$(date +%Y%m%d).sql
```

### Restore from Backup
```bash
mysql -u root -p loanlens_db < backup_20240101.sql
```

### Optimize Tables
```sql
OPTIMIZE TABLE users, loan_applications, interest_rates, loan_rules;
```

---

## Performance Considerations

1. **Indexes**: Ensure proper indexes on frequently queried columns
2. **Archiving**: Consider archiving old applications after 2 years
3. **Partitioning**: For high-volume scenarios, partition loan_applications by year
4. **Caching**: Cache interest rates and loan rules as they change infrequently

---

## Security Notes

1. **Password Encryption**: Always use BCrypt with strength 10+
2. **SQL Injection**: Use parameterized queries (JPA handles this)
3. **Sensitive Data**: Consider encrypting co_applicant_income and credit_score
4. **Audit Trail**: Add created_by, updated_by fields for audit
5. **Data Retention**: Implement GDPR-compliant data retention policies
