# LoanLens API Documentation

Base URL: `http://localhost:8080/api`

## Authentication

All protected endpoints require a Bearer token in the Authorization header:
```
Authorization: Bearer <your_jwt_token>
```

---

## Public Endpoints

### 1. Register User

**POST** `/auth/register`

Create a new user account.

**Request Body:**
```json
{
  "email": "user@example.com",
  "password": "password123",
  "fullName": "John Doe",
  "phoneNumber": "9876543210"
}
```

**Response (200 OK):**
```json
{
  "token": "eyJhbGciOiJIUzUxMiJ9...",
  "type": "Bearer",
  "userId": 1,
  "email": "user@example.com",
  "fullName": "John Doe",
  "role": "USER"
}
```

**Validations:**
- Email must be valid and unique
- Password minimum 6 characters
- Phone number must be 10 digits

---

### 2. Login

**POST** `/auth/login`

Authenticate user and get JWT token.

**Request Body:**
```json
{
  "email": "user@example.com",
  "password": "password123"
}
```

**Response (200 OK):**
```json
{
  "token": "eyJhbGciOiJIUzUxMiJ9...",
  "type": "Bearer",
  "userId": 1,
  "email": "user@example.com",
  "fullName": "John Doe",
  "role": "USER"
}
```

**Error Response (401 Unauthorized):**
```json
{
  "timestamp": "2024-01-15T10:30:00",
  "status": 401,
  "error": "Unauthorized",
  "message": "Invalid credentials"
}
```

---

## Protected Endpoints (Require Authentication)

### 3. Check Loan Eligibility

**POST** `/eligibility/check`

Calculate loan eligibility based on user inputs.

**Request Headers:**
```
Authorization: Bearer <token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "requestedAmount": 800000,
  "courseType": "ENGINEERING",
  "collegeTier": "TIER_1",
  "coApplicantIncome": 600000,
  "creditScore": 750,
  "existingLiabilities": 50000
}
```

**Course Types:**
- ENGINEERING
- MEDICAL
- MBA
- LAW
- ARTS
- SCIENCE
- COMMERCE
- OTHERS

**College Tiers:**
- TIER_1: Premier Institutions
- TIER_2: Good Institutions
- TIER_3: Standard Institutions

**Response (200 OK):**
```json
{
  "applicationId": 123,
  "eligible": true,
  "requestedAmount": 800000,
  "eligibleAmount": 2400000,
  "eligibilityScore": 78.5,
  "debtToIncomeRatio": 8.33,
  "riskLevel": "LOW",
  "riskMessage": "Your financial profile is strong. You qualify for favorable terms.",
  "recommendation": "Excellent! You're eligible for the full requested amount with competitive rates.",
  "emiOptions": [
    {
      "tenureYears": 5,
      "tenureMonths": 60,
      "monthlyEmi": 16247.95,
      "totalInterest": 174877.00,
      "totalRepayment": 974877.00,
      "interestRate": 8.50,
      "recommendation": "Shortest tenure - Save maximum on interest"
    },
    {
      "tenureYears": 7,
      "tenureMonths": 84,
      "monthlyEmi": 12304.68,
      "totalInterest": 233593.12,
      "totalRepayment": 1033593.12,
      "interestRate": 8.50,
      "recommendation": "Balanced option - Moderate EMI and interest"
    },
    {
      "tenureYears": 10,
      "tenureMonths": 120,
      "monthlyEmi": 9834.92,
      "totalInterest": 380190.40,
      "totalRepayment": 1180190.40,
      "interestRate": 8.50,
      "recommendation": "Lowest EMI - More affordable monthly payments"
    }
  ],
  "suggestedTenure": "5 years - Save on interest with higher EMI"
}
```

**Validations:**
- requestedAmount: ₹50,000 - ₹2,00,00,000
- creditScore: 300 - 900
- All amounts must be positive

---

### 4. Get My Applications

**GET** `/loans/my-applications`

Retrieve all loan applications for the authenticated user.

**Request Headers:**
```
Authorization: Bearer <token>
```

**Response (200 OK):**
```json
[
  {
    "id": 123,
    "requestedAmount": 800000,
    "courseType": "ENGINEERING",
    "collegeTier": "TIER_1",
    "coApplicantIncome": 600000,
    "creditScore": 750,
    "existingLiabilities": 50000,
    "eligibleAmount": 2400000,
    "eligibilityScore": 78.5,
    "debtToIncomeRatio": 8.33,
    "riskLevel": "LOW",
    "createdAt": "2024-01-15T10:30:00"
  }
]
```

---

### 5. Get Application by ID

**GET** `/loans/application/{id}`

Get details of a specific application.

**Request Headers:**
```
Authorization: Bearer <token>
```

**Path Parameters:**
- `id` (Long): Application ID

**Response (200 OK):**
```json
{
  "id": 123,
  "requestedAmount": 800000,
  "courseType": "ENGINEERING",
  "collegeTier": "TIER_1",
  "coApplicantIncome": 600000,
  "creditScore": 750,
  "existingLiabilities": 50000,
  "eligibleAmount": 2400000,
  "eligibilityScore": 78.5,
  "debtToIncomeRatio": 8.33,
  "riskLevel": "LOW",
  "createdAt": "2024-01-15T10:30:00"
}
```

**Error Response (403 Forbidden):**
If user doesn't own the application.

---

## Admin Endpoints (Require ADMIN Role)

### 6. Get Analytics Dashboard

**GET** `/admin/analytics`

Get analytics data for admin dashboard.

**Request Headers:**
```
Authorization: Bearer <admin_token>
```

**Response (200 OK):**
```json
{
  "totalApplications": 1250,
  "averageEligibilityScore": 68.5,
  "riskDistribution": {
    "LOW": 450,
    "MEDIUM": 600,
    "HIGH": 200
  }
}
```

---

### 7. Get All Interest Rates

**GET** `/admin/interest-rates`

Retrieve all bank interest rates.

**Request Headers:**
```
Authorization: Bearer <admin_token>
```

**Response (200 OK):**
```json
[
  {
    "id": 1,
    "bankName": "State Bank of India",
    "baseRate": 8.50,
    "minRate": 8.00,
    "maxRate": 10.00,
    "processingFeePercentage": 1.00,
    "active": true,
    "updatedAt": "2024-01-15T10:00:00",
    "specialOffer": "0% processing fee for women borrowers"
  }
]
```

---

### 8. Create Interest Rate

**POST** `/admin/interest-rates`

Add a new bank interest rate.

**Request Headers:**
```
Authorization: Bearer <admin_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "bankName": "HDFC Bank",
  "baseRate": 8.75,
  "minRate": 8.25,
  "maxRate": 10.50,
  "processingFeePercentage": 1.50,
  "active": true,
  "specialOffer": "No collateral up to 7.5 lakhs"
}
```

**Response (200 OK):**
Returns the created interest rate with ID.

---

### 9. Update Interest Rate

**PUT** `/admin/interest-rates/{id}`

Update existing interest rate.

**Path Parameters:**
- `id` (Long): Interest rate ID

**Request Body:**
```json
{
  "bankName": "HDFC Bank",
  "baseRate": 8.60,
  "minRate": 8.10,
  "maxRate": 10.25,
  "processingFeePercentage": 1.25,
  "active": true,
  "specialOffer": "Updated offer"
}
```

---

### 10. Delete Interest Rate

**DELETE** `/admin/interest-rates/{id}`

Delete an interest rate.

**Path Parameters:**
- `id` (Long): Interest rate ID

**Response (204 No Content)**

---

### 11. Get All Loan Rules

**GET** `/admin/loan-rules`

Retrieve all loan eligibility rules.

**Response (200 OK):**
```json
[
  {
    "id": 1,
    "ruleName": "Min Credit Score",
    "ruleType": "CREDIT_SCORE",
    "minValue": 300,
    "maxValue": 900,
    "threshold": 650,
    "description": "Minimum credit score required for loan approval",
    "active": true,
    "updatedAt": "2024-01-15T10:00:00"
  }
]
```

---

### 12. Create Loan Rule

**POST** `/admin/loan-rules`

Create a new loan rule.

**Request Body:**
```json
{
  "ruleName": "Max DTI Ratio",
  "ruleType": "DTI_RATIO",
  "minValue": 0,
  "maxValue": 100,
  "threshold": 40,
  "description": "Maximum debt-to-income ratio allowed",
  "active": true
}
```

---

### 13. Update Loan Rule

**PUT** `/admin/loan-rules/{id}`

Update existing loan rule.

**Path Parameters:**
- `id` (Long): Loan rule ID

---

## Error Responses

### 400 Bad Request
```json
{
  "timestamp": "2024-01-15T10:30:00",
  "status": 400,
  "error": "Bad Request",
  "message": "Validation failed",
  "errors": [
    {
      "field": "creditScore",
      "message": "Credit score must be between 300 and 900"
    }
  ]
}
```

### 401 Unauthorized
```json
{
  "timestamp": "2024-01-15T10:30:00",
  "status": 401,
  "error": "Unauthorized",
  "message": "Invalid or expired token"
}
```

### 403 Forbidden
```json
{
  "timestamp": "2024-01-15T10:30:00",
  "status": 403,
  "error": "Forbidden",
  "message": "Access denied"
}
```

### 404 Not Found
```json
{
  "timestamp": "2024-01-15T10:30:00",
  "status": 404,
  "error": "Not Found",
  "message": "Resource not found"
}
```

### 500 Internal Server Error
```json
{
  "timestamp": "2024-01-15T10:30:00",
  "status": 500,
  "error": "Internal Server Error",
  "message": "An unexpected error occurred"
}
```

---

## Rate Limiting

Currently no rate limiting is implemented. For production:
- Consider implementing rate limiting: 100 requests/minute per user
- Use Spring Cloud Gateway or Bucket4j

---

## Testing with cURL

### Register
```bash
curl -X POST http://localhost:8080/api/auth/register \
  -H "Content-Type: application/json" \
  -d '{"email":"test@example.com","password":"password123","fullName":"Test User","phoneNumber":"9876543210"}'
```

### Login
```bash
curl -X POST http://localhost:8080/api/auth/login \
  -H "Content-Type: application/json" \
  -d '{"email":"test@example.com","password":"password123"}'
```

### Check Eligibility
```bash
curl -X POST http://localhost:8080/api/eligibility/check \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -d '{"requestedAmount":800000,"courseType":"ENGINEERING","collegeTier":"TIER_1","coApplicantIncome":600000,"creditScore":750,"existingLiabilities":50000}'
```

---

## Postman Collection

Import this collection into Postman for easy API testing:

[Download Postman Collection](./LoanLens_API.postman_collection.json)

---

## WebSocket Support (Future)

For real-time updates, WebSocket support can be added:
- Endpoint: `ws://localhost:8080/ws`
- Topics: `/topic/applications`, `/topic/admin/analytics`
