package com.loanlens.service;

import com.loanlens.dto.EligibilityRequestDto;
import com.loanlens.dto.EligibilityResponseDto;
import com.loanlens.dto.EmiOptionDto;
import com.loanlens.entity.LoanApplication;
import com.loanlens.entity.User;
import com.loanlens.repository.LoanApplicationRepository;
import com.loanlens.repository.UserRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.List;

@Service
public class EligibilityService {
    
    @Autowired
    private LoanApplicationRepository loanApplicationRepository;
    
    @Autowired
    private UserRepository userRepository;
    
    @Autowired
    private EmiCalculatorService emiCalculatorService;
    
    private static final BigDecimal MIN_CREDIT_SCORE = new BigDecimal("650");
    private static final BigDecimal MAX_DTI_RATIO = new BigDecimal("40.0");
    private static final BigDecimal INCOME_MULTIPLIER = new BigDecimal("4.0");
    
    @Transactional
    public EligibilityResponseDto checkEligibility(EligibilityRequestDto request) {
        String email = SecurityContextHolder.getContext().getAuthentication().getName();
        User user = userRepository.findByEmail(email)
                .orElseThrow(() -> new RuntimeException("User not found"));
        
        // Create loan application
        LoanApplication application = new LoanApplication();
        application.setUser(user);
        application.setRequestedAmount(request.getRequestedAmount());
        application.setCourseType(request.getCourseType());
        application.setCollegeTier(request.getCollegeTier());
        application.setCoApplicantIncome(request.getCoApplicantIncome());
        application.setCreditScore(request.getCreditScore());
        application.setExistingLiabilities(request.getExistingLiabilities());
        
        // Calculate eligibility
        BigDecimal eligibilityScore = calculateEligibilityScore(application);
        BigDecimal eligibleAmount = calculateEligibleAmount(application);
        BigDecimal dtiRatio = calculateDTI(request.getExistingLiabilities(), request.getCoApplicantIncome());
        LoanApplication.RiskLevel riskLevel = determineRiskLevel(dtiRatio, request.getCreditScore());
        
        application.setEligibilityScore(eligibilityScore);
        application.setEligibleAmount(eligibleAmount);
        application.setDebtToIncomeRatio(dtiRatio);
        application.setRiskLevel(riskLevel);
        
        // Save application
        application = loanApplicationRepository.save(application);
        
        // Calculate EMI options
        BigDecimal loanAmount = eligibleAmount.min(request.getRequestedAmount());
        List<EmiOptionDto> emiOptions = calculateEmiOptions(loanAmount);
        
        // Build response
        return EligibilityResponseDto.builder()
                .applicationId(application.getId())
                .eligible(eligibilityScore.compareTo(new BigDecimal("50")) >= 0)
                .requestedAmount(request.getRequestedAmount())
                .eligibleAmount(eligibleAmount)
                .eligibilityScore(eligibilityScore)
                .debtToIncomeRatio(dtiRatio)
                .riskLevel(riskLevel)
                .riskMessage(getRiskMessage(riskLevel))
                .recommendation(getRecommendation(eligibilityScore, dtiRatio, riskLevel))
                .emiOptions(emiOptions)
                .suggestedTenure(getSuggestedTenure(dtiRatio, loanAmount, request.getCoApplicantIncome()))
                .build();
    }
    
    private BigDecimal calculateEligibilityScore(LoanApplication application) {
        BigDecimal score = new BigDecimal("0");
        
        // Credit Score Component (40%)
        BigDecimal creditScoreComponent = calculateCreditScoreComponent(application.getCreditScore());
        score = score.add(creditScoreComponent);
        
        // Income Component (30%)
        BigDecimal incomeComponent = calculateIncomeComponent(
                application.getCoApplicantIncome(),
                application.getRequestedAmount()
        );
        score = score.add(incomeComponent);
        
        // Course & College Component (20%)
        BigDecimal courseComponent = calculateCourseComponent(
                application.getCourseType(),
                application.getCollegeTier()
        );
        score = score.add(courseComponent);
        
        // DTI Component (10%)
        BigDecimal dtiComponent = calculateDTIComponent(
                application.getExistingLiabilities(),
                application.getCoApplicantIncome()
        );
        score = score.add(dtiComponent);
        
        return score.setScale(2, RoundingMode.HALF_UP);
    }
    
    private BigDecimal calculateCreditScoreComponent(Integer creditScore) {
        if (creditScore >= 750) return new BigDecimal("40");
        if (creditScore >= 700) return new BigDecimal("35");
        if (creditScore >= 650) return new BigDecimal("25");
        if (creditScore >= 600) return new BigDecimal("15");
        return new BigDecimal("5");
    }
    
    private BigDecimal calculateIncomeComponent(BigDecimal income, BigDecimal requestedAmount) {
        BigDecimal affordableAmount = income.multiply(INCOME_MULTIPLIER);
        if (requestedAmount.compareTo(affordableAmount) <= 0) {
            return new BigDecimal("30");
        }
        BigDecimal ratio = affordableAmount.divide(requestedAmount, 2, RoundingMode.HALF_UP);
        return new BigDecimal("30").multiply(ratio);
    }
    
    private BigDecimal calculateCourseComponent(LoanApplication.CourseType courseType,
                                                 LoanApplication.CollegeTier collegeTier) {
        double courseRisk = courseType.getRiskFactor();
        double tierMultiplier = collegeTier.getMultiplier();
        double component = 20 * courseRisk * tierMultiplier;
        return new BigDecimal(component).setScale(2, RoundingMode.HALF_UP);
    }
    
    private BigDecimal calculateDTIComponent(BigDecimal liabilities, BigDecimal income) {
        if (income.compareTo(BigDecimal.ZERO) == 0) return new BigDecimal("0");
        
        BigDecimal dti = liabilities.divide(income, 4, RoundingMode.HALF_UP)
                .multiply(new BigDecimal("100"));
        
        if (dti.compareTo(new BigDecimal("20")) <= 0) return new BigDecimal("10");
        if (dti.compareTo(new BigDecimal("30")) <= 0) return new BigDecimal("7");
        if (dti.compareTo(new BigDecimal("40")) <= 0) return new BigDecimal("4");
        return new BigDecimal("0");
    }
    
    private BigDecimal calculateEligibleAmount(LoanApplication application) {
        BigDecimal maxByIncome = application.getCoApplicantIncome().multiply(INCOME_MULTIPLIER);
        BigDecimal maxByCredit = getMaxLoanByCredit(application.getCreditScore());
        BigDecimal courseAdjusted = maxByIncome.multiply(
                new BigDecimal(application.getCourseType().getRiskFactor())
        );
        BigDecimal tierAdjusted = courseAdjusted.multiply(
                new BigDecimal(application.getCollegeTier().getMultiplier())
        );
        
        return tierAdjusted.min(maxByCredit).setScale(2, RoundingMode.HALF_UP);
    }
    
    private BigDecimal getMaxLoanByCredit(Integer creditScore) {
        if (creditScore >= 750) return new BigDecimal("2000000"); // 20 lakhs
        if (creditScore >= 700) return new BigDecimal("1500000"); // 15 lakhs
        if (creditScore >= 650) return new BigDecimal("1000000"); // 10 lakhs
        return new BigDecimal("500000"); // 5 lakhs
    }
    
    private BigDecimal calculateDTI(BigDecimal liabilities, BigDecimal income) {
        if (income.compareTo(BigDecimal.ZERO) == 0) return new BigDecimal("100");
        return liabilities.divide(income, 4, RoundingMode.HALF_UP)
                .multiply(new BigDecimal("100"))
                .setScale(2, RoundingMode.HALF_UP);
    }
    
    private LoanApplication.RiskLevel determineRiskLevel(BigDecimal dtiRatio, Integer creditScore) {
        if (dtiRatio.compareTo(new BigDecimal("40")) > 0 || creditScore < 600) {
            return LoanApplication.RiskLevel.HIGH;
        }
        if (dtiRatio.compareTo(new BigDecimal("30")) > 0 || creditScore < 700) {
            return LoanApplication.RiskLevel.MEDIUM;
        }
        return LoanApplication.RiskLevel.LOW;
    }
    
    private List<EmiOptionDto> calculateEmiOptions(BigDecimal loanAmount) {
        List<EmiOptionDto> options = new ArrayList<>();
        int[] tenures = {5, 7, 10};
        
        for (int tenure : tenures) {
            EmiOptionDto option = emiCalculatorService.calculateEmi(loanAmount, tenure);
            options.add(option);
        }
        
        return options;
    }
    
    private String getRiskMessage(LoanApplication.RiskLevel riskLevel) {
        return switch (riskLevel) {
            case LOW -> "Your financial profile is strong. You qualify for favorable terms.";
            case MEDIUM -> "Your profile is moderate. Consider reducing existing debts for better terms.";
            case HIGH -> "Your profile shows high risk. Focus on improving credit score and reducing DTI.";
        };
    }
    
    private String getRecommendation(BigDecimal eligibilityScore, BigDecimal dtiRatio, 
                                     LoanApplication.RiskLevel riskLevel) {
        if (eligibilityScore.compareTo(new BigDecimal("70")) >= 0) {
            return "Excellent! You're eligible for the full requested amount with competitive rates.";
        }
        if (eligibilityScore.compareTo(new BigDecimal("50")) >= 0) {
            return "You're eligible for a loan, but consider improving your credit score for better rates.";
        }
        return "Consider improving your credit score and reducing existing debts before applying.";
    }
    
    private String getSuggestedTenure(BigDecimal dtiRatio, BigDecimal loanAmount, BigDecimal income) {
        BigDecimal monthlyIncome = income.divide(new BigDecimal("12"), 2, RoundingMode.HALF_UP);
        
        if (dtiRatio.compareTo(new BigDecimal("30")) > 0) {
            return "10 years - Lower EMI for better affordability";
        }
        if (dtiRatio.compareTo(new BigDecimal("20")) > 0) {
            return "7 years - Balanced approach";
        }
        return "5 years - Save on interest with higher EMI";
    }
}
