package com.loanlens.controller;

import com.loanlens.entity.InterestRate;
import com.loanlens.entity.LoanRule;
import com.loanlens.repository.InterestRateRepository;
import com.loanlens.repository.LoanApplicationRepository;
import com.loanlens.repository.LoanRuleRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/api/admin")
@PreAuthorize("hasRole('ADMIN')")
public class AdminController {
    
    @Autowired
    private LoanApplicationRepository loanApplicationRepository;
    
    @Autowired
    private InterestRateRepository interestRateRepository;
    
    @Autowired
    private LoanRuleRepository loanRuleRepository;
    
    @GetMapping("/analytics")
    public ResponseEntity<Map<String, Object>> getAnalytics() {
        Map<String, Object> analytics = new HashMap<>();
        
        // Total applications
        Long totalApplications = loanApplicationRepository.countTotalApplications();
        analytics.put("totalApplications", totalApplications);
        
        // Average eligibility score
        Double avgScore = loanApplicationRepository.getAverageEligibilityScore();
        analytics.put("averageEligibilityScore", avgScore);
        
        // Risk level distribution
        List<Object[]> riskDistribution = loanApplicationRepository.getRiskLevelDistribution();
        Map<String, Long> riskMap = new HashMap<>();
        for (Object[] row : riskDistribution) {
            riskMap.put(row[0].toString(), (Long) row[1]);
        }
        analytics.put("riskDistribution", riskMap);
        
        return ResponseEntity.ok(analytics);
    }
    
    @GetMapping("/interest-rates")
    public ResponseEntity<List<InterestRate>> getAllInterestRates() {
        return ResponseEntity.ok(interestRateRepository.findAll());
    }
    
    @PostMapping("/interest-rates")
    public ResponseEntity<InterestRate> createInterestRate(@RequestBody InterestRate interestRate) {
        return ResponseEntity.ok(interestRateRepository.save(interestRate));
    }
    
    @PutMapping("/interest-rates/{id}")
    public ResponseEntity<InterestRate> updateInterestRate(
            @PathVariable Long id,
            @RequestBody InterestRate interestRate) {
        InterestRate existing = interestRateRepository.findById(id)
                .orElseThrow(() -> new RuntimeException("Interest rate not found"));
        
        existing.setBankName(interestRate.getBankName());
        existing.setBaseRate(interestRate.getBaseRate());
        existing.setMinRate(interestRate.getMinRate());
        existing.setMaxRate(interestRate.getMaxRate());
        existing.setProcessingFeePercentage(interestRate.getProcessingFeePercentage());
        existing.setActive(interestRate.getActive());
        existing.setSpecialOffer(interestRate.getSpecialOffer());
        
        return ResponseEntity.ok(interestRateRepository.save(existing));
    }
    
    @DeleteMapping("/interest-rates/{id}")
    public ResponseEntity<Void> deleteInterestRate(@PathVariable Long id) {
        interestRateRepository.deleteById(id);
        return ResponseEntity.noContent().build();
    }
    
    @GetMapping("/loan-rules")
    public ResponseEntity<List<LoanRule>> getAllLoanRules() {
        return ResponseEntity.ok(loanRuleRepository.findAll());
    }
    
    @PostMapping("/loan-rules")
    public ResponseEntity<LoanRule> createLoanRule(@RequestBody LoanRule loanRule) {
        return ResponseEntity.ok(loanRuleRepository.save(loanRule));
    }
    
    @PutMapping("/loan-rules/{id}")
    public ResponseEntity<LoanRule> updateLoanRule(
            @PathVariable Long id,
            @RequestBody LoanRule loanRule) {
        LoanRule existing = loanRuleRepository.findById(id)
                .orElseThrow(() -> new RuntimeException("Loan rule not found"));
        
        existing.setRuleName(loanRule.getRuleName());
        existing.setRuleType(loanRule.getRuleType());
        existing.setMinValue(loanRule.getMinValue());
        existing.setMaxValue(loanRule.getMaxValue());
        existing.setThreshold(loanRule.getThreshold());
        existing.setDescription(loanRule.getDescription());
        existing.setActive(loanRule.getActive());
        
        return ResponseEntity.ok(loanRuleRepository.save(existing));
    }
}
