# LoanLens - Complete Implementation Guide

## Backend Implementation (Spring Boot)

### Setup Steps

1. **Database Setup**
```sql
CREATE DATABASE loanlens_db;
USE loanlens_db;

-- Insert default admin user (password: admin123)
INSERT INTO users (email, password, full_name, phone_number, role, active, created_at, updated_at)
VALUES ('admin@loanlens.com', '$2a$10$xwVzYiAXVCKxBQ4H6U3lTe9bQrMwKq8v4GNh8kF2PzG7xYzI5jKoK', 'Admin User', '9999999999', 'ADMIN', true, NOW(), NOW());

-- Insert default interest rates
INSERT INTO interest_rates (bank_name, base_rate, min_rate, max_rate, processing_fee_percentage, active, updated_at)
VALUES 
('State Bank of India', 8.50, 8.00, 10.00, 1.00, true, NOW()),
('HDFC Bank', 8.75, 8.25, 10.50, 1.50, true, NOW()),
('ICICI Bank', 8.60, 8.10, 10.25, 1.25, true, NOW());

-- Insert default loan rules
INSERT INTO loan_rules (rule_name, rule_type, min_value, max_value, threshold, description, active, updated_at)
VALUES 
('Min Credit Score', 'CREDIT_SCORE', 300, 900, 650, 'Minimum credit score required for loan approval', true, NOW()),
('Max DTI Ratio', 'DTI_RATIO', 0, 100, 40, 'Maximum debt-to-income ratio allowed', true, NOW()),
('Max Loan Amount', 'MAX_LOAN', 50000, 20000000, NULL, 'Maximum loan amount per application', true, NOW());
```

2. **Run Backend**
```bash
cd backend
mvn clean install
mvn spring-boot:run
```

Backend will be available at: http://localhost:8080

### API Testing with cURL

```bash
# Register User
curl -X POST http://localhost:8080/api/auth/register \
  -H "Content-Type: application/json" \
  -d '{
    "email": "test@example.com",
    "password": "password123",
    "fullName": "Test User",
    "phoneNumber": "9876543210"
  }'

# Login
curl -X POST http://localhost:8080/api/auth/login \
  -H "Content-Type: application/json" \
  -d '{
    "email": "test@example.com",
    "password": "password123"
  }'

# Check Eligibility (use token from login response)
curl -X POST http://localhost:8080/api/eligibility/check \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -d '{
    "requestedAmount": 800000,
    "courseType": "ENGINEERING",
    "collegeTier": "TIER_1",
    "coApplicantIncome": 600000,
    "creditScore": 750,
    "existingLiabilities": 50000
  }'
```

## Frontend Implementation (React)

### Remaining React Components

#### 1. Register.jsx
```jsx
import React, { useState } from 'react';
import { useNavigate, Link } from 'react-router-dom';
import { authAPI } from '../services/api';

const Register = () => {
  const navigate = useNavigate();
  const [formData, setFormData] = useState({
    email: '',
    password: '',
    fullName: '',
    phoneNumber: '',
  });
  const [error, setError] = useState('');
  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);
    try {
      await authAPI.register(formData);
      navigate('/login');
    } catch (err) {
      setError(err.response?.data?.message || 'Registration failed');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 flex items-center justify-center py-12 px-4">
      <div className="max-w-md w-full bg-white p-10 rounded-xl shadow-lg">
        <h2 className="text-3xl font-extrabold text-gray-900 text-center">Create Account</h2>
        {error && <div className="mt-4 bg-red-50 border border-red-400 text-red-700 px-4 py-3 rounded">{error}</div>}
        <form onSubmit={handleSubmit} className="mt-8 space-y-4">
          <input
            type="text"
            required
            placeholder="Full Name"
            className="w-full px-3 py-2 border border-gray-300 rounded-md"
            value={formData.fullName}
            onChange={(e) => setFormData({...formData, fullName: e.target.value})}
          />
          <input
            type="email"
            required
            placeholder="Email"
            className="w-full px-3 py-2 border border-gray-300 rounded-md"
            value={formData.email}
            onChange={(e) => setFormData({...formData, email: e.target.value})}
          />
          <input
            type="tel"
            placeholder="Phone Number (10 digits)"
            pattern="[0-9]{10}"
            className="w-full px-3 py-2 border border-gray-300 rounded-md"
            value={formData.phoneNumber}
            onChange={(e) => setFormData({...formData, phoneNumber: e.target.value})}
          />
          <input
            type="password"
            required
            placeholder="Password (min 6 characters)"
            minLength="6"
            className="w-full px-3 py-2 border border-gray-300 rounded-md"
            value={formData.password}
            onChange={(e) => setFormData({...formData, password: e.target.value})}
          />
          <button
            type="submit"
            disabled={loading}
            className="w-full py-2 px-4 bg-indigo-600 hover:bg-indigo-700 text-white font-medium rounded-md disabled:bg-indigo-400"
          >
            {loading ? 'Creating Account...' : 'Register'}
          </button>
          <p className="text-center text-sm text-gray-600">
            Already have an account? <Link to="/login" className="text-indigo-600 hover:text-indigo-500">Login</Link>
          </p>
        </form>
      </div>
    </div>
  );
};

export default Register;
```

#### 2. Dashboard.jsx (Main Application Page)
```jsx
import React, { useState } from 'react';
import { useAuth } from '../context/AuthContext';
import EligibilityForm from '../components/EligibilityForm';
import EligibilityResult from '../components/EligibilityResult';
import ApplicationHistory from '../components/ApplicationHistory';

const Dashboard = () => {
  const { user, logout } = useAuth();
  const [result, setResult] = useState(null);
  const [activeTab, setActiveTab] = useState('check');

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Navigation */}
      <nav className="bg-white shadow-sm">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between h-16">
            <div className="flex items-center">
              <h1 className="text-2xl font-bold text-indigo-600">🎓 LoanLens</h1>
            </div>
            <div className="flex items-center space-x-4">
              <span className="text-gray-700">Welcome, {user?.fullName}</span>
              <button
                onClick={logout}
                className="px-4 py-2 text-sm text-red-600 hover:text-red-700"
              >
                Logout
              </button>
            </div>
          </div>
        </div>
      </nav>

      {/* Main Content */}
      <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
        {/* Tabs */}
        <div className="bg-white shadow rounded-lg mb-6">
          <div className="border-b border-gray-200">
            <nav className="-mb-px flex">
              <button
                onClick={() => setActiveTab('check')}
                className={`${
                  activeTab === 'check'
                    ? 'border-indigo-500 text-indigo-600'
                    : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                } whitespace-nowrap py-4 px-6 border-b-2 font-medium text-sm`}
              >
                Check Eligibility
              </button>
              <button
                onClick={() => setActiveTab('history')}
                className={`${
                  activeTab === 'history'
                    ? 'border-indigo-500 text-indigo-600'
                    : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                } whitespace-nowrap py-4 px-6 border-b-2 font-medium text-sm`}
              >
                My Applications
              </button>
            </nav>
          </div>

          <div className="p-6">
            {activeTab === 'check' && (
              <div>
                <EligibilityForm onResult={setResult} />
                {result && <EligibilityResult result={result} />}
              </div>
            )}
            {activeTab === 'history' && <ApplicationHistory />}
          </div>
        </div>
      </div>
    </div>
  );
};

export default Dashboard;
```

#### 3. EligibilityForm.jsx
```jsx
import React, { useState } from 'react';
import { eligibilityAPI } from '../services/api';

const EligibilityForm = ({ onResult }) => {
  const [formData, setFormData] = useState({
    requestedAmount: '',
    courseType: '',
    collegeTier: '',
    coApplicantIncome: '',
    creditScore: '',
    existingLiabilities: '',
  });
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');

  const courseTypes = [
    { value: 'ENGINEERING', label: 'Engineering' },
    { value: 'MEDICAL', label: 'Medical' },
    { value: 'MBA', label: 'MBA' },
    { value: 'LAW', label: 'Law' },
    { value: 'ARTS', label: 'Arts & Humanities' },
    { value: 'SCIENCE', label: 'Science' },
    { value: 'COMMERCE', label: 'Commerce' },
    { value: 'OTHERS', label: 'Others' },
  ];

  const collegeTiers = [
    { value: 'TIER_1', label: 'Tier 1 - Premier Institutions (IIT, AIIMS, IIM, NLU)' },
    { value: 'TIER_2', label: 'Tier 2 - Good Institutions (NIT, State Universities)' },
    { value: 'TIER_3', label: 'Tier 3 - Standard Institutions' },
  ];

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);
    setError('');

    try {
      const response = await eligibilityAPI.checkEligibility({
        ...formData,
        requestedAmount: parseFloat(formData.requestedAmount),
        coApplicantIncome: parseFloat(formData.coApplicantIncome),
        creditScore: parseInt(formData.creditScore),
        existingLiabilities: parseFloat(formData.existingLiabilities),
      });
      onResult(response.data);
    } catch (err) {
      setError(err.response?.data?.message || 'Failed to check eligibility');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="bg-white rounded-lg shadow p-6">
      <h2 className="text-2xl font-bold text-gray-900 mb-6">Check Loan Eligibility</h2>
      
      {error && (
        <div className="mb-4 bg-red-50 border border-red-400 text-red-700 px-4 py-3 rounded">
          {error}
        </div>
      )}

      <form onSubmit={handleSubmit} className="space-y-6">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Requested Loan Amount (₹)
            </label>
            <input
              type="number"
              required
              min="50000"
              max="20000000"
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-indigo-500 focus:border-indigo-500"
              value={formData.requestedAmount}
              onChange={(e) => setFormData({...formData, requestedAmount: e.target.value})}
              placeholder="e.g., 800000"
            />
            <p className="mt-1 text-sm text-gray-500">Min: ₹50,000 | Max: ₹2,00,00,000</p>
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Course Type
            </label>
            <select
              required
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-indigo-500 focus:border-indigo-500"
              value={formData.courseType}
              onChange={(e) => setFormData({...formData, courseType: e.target.value})}
            >
              <option value="">Select Course Type</option>
              {courseTypes.map((type) => (
                <option key={type.value} value={type.value}>{type.label}</option>
              ))}
            </select>
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              College Tier
            </label>
            <select
              required
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-indigo-500 focus:border-indigo-500"
              value={formData.collegeTier}
              onChange={(e) => setFormData({...formData, collegeTier: e.target.value})}
            >
              <option value="">Select College Tier</option>
              {collegeTiers.map((tier) => (
                <option key={tier.value} value={tier.value}>{tier.label}</option>
              ))}
            </select>
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Co-Applicant Annual Income (₹)
            </label>
            <input
              type="number"
              required
              min="0"
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-indigo-500 focus:border-indigo-500"
              value={formData.coApplicantIncome}
              onChange={(e) => setFormData({...formData, coApplicantIncome: e.target.value})}
              placeholder="e.g., 600000"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Credit Score
            </label>
            <input
              type="number"
              required
              min="300"
              max="900"
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-indigo-500 focus:border-indigo-500"
              value={formData.creditScore}
              onChange={(e) => setFormData({...formData, creditScore: e.target.value})}
              placeholder="e.g., 750"
            />
            <p className="mt-1 text-sm text-gray-500">Range: 300 - 900</p>
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Existing Liabilities (₹)
            </label>
            <input
              type="number"
              required
              min="0"
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-indigo-500 focus:border-indigo-500"
              value={formData.existingLiabilities}
              onChange={(e) => setFormData({...formData, existingLiabilities: e.target.value})}
              placeholder="e.g., 50000"
            />
            <p className="mt-1 text-sm text-gray-500">Total of all existing EMIs</p>
          </div>
        </div>

        <div>
          <button
            type="submit"
            disabled={loading}
            className="w-full py-3 px-4 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 disabled:bg-indigo-400"
          >
            {loading ? 'Calculating...' : 'Check Eligibility'}
          </button>
        </div>
      </form>
    </div>
  );
};

export default EligibilityForm;
```

#### 4. EligibilityResult.jsx
```jsx
import React from 'react';
import { Chart as ChartJS, ArcElement, Tooltip, Legend, CategoryScale, LinearScale, BarElement } from 'chart.js';
import { Pie, Bar } from 'react-chartjs-2';

ChartJS.register(ArcElement, Tooltip, Legend, CategoryScale, LinearScale, BarElement);

const EligibilityResult = ({ result }) => {
  const getRiskColor = (riskLevel) => {
    switch (riskLevel) {
      case 'LOW': return 'bg-green-100 text-green-800 border-green-300';
      case 'MEDIUM': return 'bg-yellow-100 text-yellow-800 border-yellow-300';
      case 'HIGH': return 'bg-red-100 text-red-800 border-red-300';
      default: return 'bg-gray-100 text-gray-800 border-gray-300';
    }
  };

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      maximumFractionDigits: 0,
    }).format(amount);
  };

  const emiChartData = {
    labels: result.emiOptions.map(opt => `${opt.tenureYears} Years`),
    datasets: [
      {
        label: 'Monthly EMI (₹)',
        data: result.emiOptions.map(opt => opt.monthlyEmi),
        backgroundColor: 'rgba(99, 102, 241, 0.5)',
        borderColor: 'rgb(99, 102, 241)',
        borderWidth: 1,
      },
    ],
  };

  return (
    <div className="mt-8 space-y-6">
      {/* Eligibility Status */}
      <div className={`rounded-lg border-2 p-6 ${result.eligible ? 'bg-green-50 border-green-300' : 'bg-red-50 border-red-300'}`}>
        <h3 className="text-2xl font-bold mb-2">
          {result.eligible ? '✅ Congratulations! You are Eligible' : '❌ Not Eligible'}
        </h3>
        <p className="text-lg">{result.recommendation}</p>
      </div>

      {/* Key Metrics */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <div className="bg-white rounded-lg shadow p-6">
          <div className="text-sm text-gray-600">Eligibility Score</div>
          <div className="text-3xl font-bold text-indigo-600">{result.eligibilityScore}/100</div>
        </div>
        <div className="bg-white rounded-lg shadow p-6">
          <div className="text-sm text-gray-600">Eligible Amount</div>
          <div className="text-3xl font-bold text-green-600">{formatCurrency(result.eligibleAmount)}</div>
        </div>
        <div className="bg-white rounded-lg shadow p-6">
          <div className="text-sm text-gray-600">DTI Ratio</div>
          <div className="text-3xl font-bold text-orange-600">{result.debtToIncomeRatio}%</div>
        </div>
      </div>

      {/* Risk Indicator */}
      <div className={`rounded-lg border-2 p-6 ${getRiskColor(result.riskLevel)}`}>
        <h4 className="text-lg font-semibold mb-2">Risk Level: {result.riskLevel}</h4>
        <p>{result.riskMessage}</p>
      </div>

      {/* EMI Options */}
      <div className="bg-white rounded-lg shadow p-6">
        <h3 className="text-xl font-bold mb-4">EMI Options</h3>
        <div className="mb-6">
          <Bar data={emiChartData} options={{ responsive: true, plugins: { legend: { display: true } } }} />
        </div>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          {result.emiOptions.map((option, index) => (
            <div key={index} className="border rounded-lg p-4 hover:shadow-md transition">
              <div className="text-lg font-semibold text-indigo-600 mb-2">
                {option.tenureYears} Years
              </div>
              <div className="space-y-2 text-sm">
                <div className="flex justify-between">
                  <span className="text-gray-600">Monthly EMI:</span>
                  <span className="font-semibold">{formatCurrency(option.monthlyEmi)}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-600">Total Interest:</span>
                  <span className="font-semibold">{formatCurrency(option.totalInterest)}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-600">Total Repayment:</span>
                  <span className="font-semibold">{formatCurrency(option.totalRepayment)}</span>
                </div>
                <div className="mt-2 pt-2 border-t">
                  <p className="text-xs text-gray-500 italic">{option.recommendation}</p>
                </div>
              </div>
            </div>
          ))}
        </div>
        <div className="mt-4 bg-blue-50 border border-blue-200 rounded p-3">
          <p className="text-sm text-blue-800">
            <strong>💡 Suggested Tenure:</strong> {result.suggestedTenure}
          </p>
        </div>
      </div>
    </div>
  );
};

export default EligibilityResult;
```

#### 5. ApplicationHistory.jsx
```jsx
import React, { useEffect, useState } from 'react';
import { loanAPI } from '../services/api';

const ApplicationHistory = () => {
  const [applications, setApplications] = useState([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchApplications();
  }, []);

  const fetchApplications = async () => {
    try {
      const response = await loanAPI.getMyApplications();
      setApplications(response.data);
    } catch (error) {
      console.error('Failed to fetch applications', error);
    } finally {
      setLoading(false);
    }
  };

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      maximumFractionDigits: 0,
    }).format(amount);
  };

  const formatDate = (dateString) => {
    return new Date(dateString).toLocaleDateString('en-IN', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  if (loading) {
    return <div className="text-center py-8">Loading...</div>;
  }

  if (applications.length === 0) {
    return (
      <div className="text-center py-8">
        <p className="text-gray-500">No applications found. Check your eligibility to get started!</p>
      </div>
    );
  }

  return (
    <div className="space-y-4">
      <h2 className="text-2xl font-bold text-gray-900 mb-4">Your Loan Applications</h2>
      {applications.map((app) => (
        <div key={app.id} className="bg-white border rounded-lg shadow-sm p-6 hover:shadow-md transition">
          <div className="flex justify-between items-start mb-4">
            <div>
              <h3 className="text-lg font-semibold text-gray-900">
                Application #{app.id}
              </h3>
              <p className="text-sm text-gray-500">{formatDate(app.createdAt)}</p>
            </div>
            <span className={`px-3 py-1 rounded-full text-sm font-medium ${
              app.riskLevel === 'LOW' ? 'bg-green-100 text-green-800' :
              app.riskLevel === 'MEDIUM' ? 'bg-yellow-100 text-yellow-800' :
              'bg-red-100 text-red-800'
            }`}>
              {app.riskLevel} Risk
            </span>
          </div>
          
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
            <div>
              <div className="text-sm text-gray-600">Requested Amount</div>
              <div className="font-semibold">{formatCurrency(app.requestedAmount)}</div>
            </div>
            <div>
              <div className="text-sm text-gray-600">Eligible Amount</div>
              <div className="font-semibold text-green-600">{formatCurrency(app.eligibleAmount)}</div>
            </div>
            <div>
              <div className="text-sm text-gray-600">Course</div>
              <div className="font-semibold">{app.courseType}</div>
            </div>
            <div>
              <div className="text-sm text-gray-600">Credit Score</div>
              <div className="font-semibold">{app.creditScore}</div>
            </div>
          </div>
        </div>
      ))}
    </div>
  );
};

export default ApplicationHistory;
```

#### 6. App.jsx (Main Router)
```jsx
import React from 'react';
import { BrowserRouter as Router, Routes, Route, Navigate } from 'react-router-dom';
import { AuthProvider, useAuth } from './context/AuthContext';
import Login from './pages/Login';
import Register from './pages/Register';
import Dashboard from './pages/Dashboard';

const PrivateRoute = ({ children }) => {
  const { user, loading } = useAuth();
  
  if (loading) {
    return <div className="min-h-screen flex items-center justify-center">Loading...</div>;
  }
  
  return user ? children : <Navigate to="/login" />;
};

const App = () => {
  return (
    <AuthProvider>
      <Router>
        <Routes>
          <Route path="/login" element={<Login />} />
          <Route path="/register" element={<Register />} />
          <Route path="/dashboard" element={
            <PrivateRoute>
              <Dashboard />
            </PrivateRoute>
          } />
          <Route path="/" element={<Navigate to="/dashboard" />} />
        </Routes>
      </Router>
    </AuthProvider>
  );
};

export default App;
```

#### 7. index.jsx
```jsx
import React from 'react';
import ReactDOM from 'react-dom/client';
import './index.css';
import App from './App';

ReactDOM.createRoot(document.getElementById('root')).render(
  <React.StrictMode>
    <App />
  </React.StrictMode>
);
```

#### 8. index.css
```css
@tailwind base;
@tailwind components;
@tailwind utilities;

body {
  margin: 0;
  font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Roboto', 'Oxygen',
    'Ubuntu', 'Cantarell', 'Fira Sans', 'Droid Sans', 'Helvetica Neue',
    sans-serif;
  -webkit-font-smoothing: antialiased;
  -moz-osx-font-smoothing: grayscale;
}
```

#### 9. index.html
```html
<!DOCTYPE html>
<html lang="en">
  <head>
    <meta charset="UTF-8" />
    <link rel="icon" type="image/svg+xml" href="/vite.svg" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0" />
    <title>LoanLens - Student Loan Eligibility Checker</title>
  </head>
  <body>
    <div id="root"></div>
    <script type="module" src="/src/index.jsx"></script>
  </body>
</html>
```

### Frontend Setup

1. **Install Dependencies**
```bash
cd frontend
npm install
```

2. **Create .env file**
```
VITE_API_BASE_URL=http://localhost:8080/api
```

3. **Run Development Server**
```bash
npm run dev
```

Frontend will be available at: http://localhost:3000

## Testing the Complete Application

### User Flow Testing

1. **Register a New User**
   - Navigate to http://localhost:3000/register
   - Fill in: Name, Email, Phone, Password
   - Click Register

2. **Login**
   - Navigate to http://localhost:3000/login
   - Enter credentials
   - You'll be redirected to Dashboard

3. **Check Loan Eligibility**
   - Fill in the eligibility form:
     * Requested Amount: ₹8,00,000
     * Course: Engineering
     * College Tier: Tier 1
     * Income: ₹6,00,000
     * Credit Score: 750
     * Existing Liabilities: ₹50,000
   - Click "Check Eligibility"
   - View results with EMI options

4. **View Application History**
   - Click "My Applications" tab
   - See all your previous applications

## Key Features Demonstration

### Eligibility Calculation Logic

The system calculates eligibility based on:

1. **Credit Score (40% weight)**
   - 750+: 40 points
   - 700-749: 35 points
   - 650-699: 25 points
   - 600-649: 15 points
   - <600: 5 points

2. **Income (30% weight)**
   - Can afford 4x income as loan
   - Proportional scoring

3. **Course & College (20% weight)**
   - Course risk factors (Engineering: 0.85, Medical: 0.90, etc.)
   - College tier multipliers (Tier 1: 1.0, Tier 2: 0.85, Tier 3: 0.70)

4. **DTI Ratio (10% weight)**
   - <20%: 10 points
   - 20-30%: 7 points
   - 30-40%: 4 points
   - >40%: 0 points

### Risk Level Determination

- **Low Risk (Green)**: DTI < 30% AND Credit Score ≥ 700
- **Medium Risk (Yellow)**: DTI 30-40% OR Credit Score 600-699
- **High Risk (Red)**: DTI > 40% OR Credit Score < 600

### EMI Calculation Formula

```
EMI = [P × R × (1+R)^N] / [(1+R)^N - 1]

Where:
P = Principal loan amount
R = Monthly interest rate (Annual rate / 12 / 100)
N = Tenure in months
```

## Production Deployment

### Backend Deployment

1. **Build JAR**
```bash
mvn clean package -DskipTests
```

2. **Run in Production**
```bash
java -jar target/loanlens-backend-1.0.0.jar --spring.profiles.active=prod
```

3. **Create production application.properties**
```properties
spring.datasource.url=jdbc:mysql://your-prod-db:3306/loanlens_db
spring.datasource.username=${DB_USERNAME}
spring.datasource.password=${DB_PASSWORD}
jwt.secret=${JWT_SECRET}
cors.allowed-origins=${ALLOWED_ORIGINS}
```

### Frontend Deployment

1. **Build for Production**
```bash
npm run build
```

2. **Deploy dist/ folder** to:
   - Netlify
   - Vercel
   - AWS S3 + CloudFront
   - Nginx server

### Environment Variables

**Backend (.env or system vars):**
- DB_USERNAME
- DB_PASSWORD
- JWT_SECRET

**Frontend (.env.production):**
- VITE_API_BASE_URL=https://your-api-domain.com/api

## Troubleshooting

### Common Issues

1. **CORS Errors**
   - Check `cors.allowed-origins` in application.properties
   - Ensure frontend URL is whitelisted

2. **JWT Token Issues**
   - Verify JWT secret is properly configured
   - Check token expiration time

3. **Database Connection**
   - Verify MySQL is running
   - Check credentials in application.properties

4. **Build Failures**
   - Ensure Java 17+ is installed
   - Ensure Node.js 16+ is installed
   - Clear Maven cache: `mvn clean`
   - Clear npm cache: `npm cache clean --force`

## Advanced Features (Future Enhancements)

1. **Loan Comparison**
   - Compare multiple banks
   - Show best offers

2. **Document Upload**
   - Upload income proof
   - Upload college admission letter

3. **Email Notifications**
   - Send eligibility results via email
   - Reminder notifications

4. **Admin Dashboard**
   - View analytics charts
   - Manage interest rates
   - Configure loan rules

5. **Reports Generation**
   - Download PDF reports
   - Export application data

## Contact & Support

For issues or questions:
- Email: support@loanlens.com
- GitHub: https://github.com/yourrepo/loanlens

---

**Happy Coding! 🚀**
