package com.loanlens.service;

import com.loanlens.dto.EmiOptionDto;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;

@Service
public class EmiCalculatorService {
    
    // Base interest rate - can be fetched from InterestRate table
    private static final BigDecimal BASE_RATE = new BigDecimal("8.5");
    
    public EmiOptionDto calculateEmi(BigDecimal principal, int tenureYears) {
        int tenureMonths = tenureYears * 12;
        BigDecimal monthlyRate = BASE_RATE.divide(new BigDecimal("1200"), 6, RoundingMode.HALF_UP);
        
        // EMI = [P x R x (1+R)^N] / [(1+R)^N-1]
        BigDecimal onePlusR = BigDecimal.ONE.add(monthlyRate);
        BigDecimal onePlusRPowN = onePlusR.pow(tenureMonths);
        
        BigDecimal numerator = principal.multiply(monthlyRate).multiply(onePlusRPowN);
        BigDecimal denominator = onePlusRPowN.subtract(BigDecimal.ONE);
        
        BigDecimal monthlyEmi = numerator.divide(denominator, 2, RoundingMode.HALF_UP);
        BigDecimal totalRepayment = monthlyEmi.multiply(new BigDecimal(tenureMonths));
        BigDecimal totalInterest = totalRepayment.subtract(principal);
        
        String recommendation = getRecommendation(tenureYears, monthlyEmi, totalInterest);
        
        return EmiOptionDto.builder()
                .tenureYears(tenureYears)
                .tenureMonths(tenureMonths)
                .monthlyEmi(monthlyEmi)
                .totalInterest(totalInterest)
                .totalRepayment(totalRepayment)
                .interestRate(BASE_RATE)
                .recommendation(recommendation)
                .build();
    }
    
    private String getRecommendation(int tenure, BigDecimal emi, BigDecimal interest) {
        if (tenure == 5) {
            return "Shortest tenure - Save maximum on interest";
        } else if (tenure == 7) {
            return "Balanced option - Moderate EMI and interest";
        } else {
            return "Lowest EMI - More affordable monthly payments";
        }
    }
}
