package com.loanlens.entity;

import jakarta.persistence.*;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.hibernate.annotations.CreationTimestamp;

import java.math.BigDecimal;
import java.time.LocalDateTime;

@Entity
@Table(name = "loan_applications")
@Data
@NoArgsConstructor
@AllArgsConstructor
public class LoanApplication {
    
    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    private Long id;
    
    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "user_id", nullable = false)
    private User user;
    
    @Column(nullable = false, precision = 12, scale = 2)
    private BigDecimal requestedAmount;
    
    @Column(nullable = false)
    @Enumerated(EnumType.STRING)
    private CourseType courseType;
    
    @Column(nullable = false)
    @Enumerated(EnumType.STRING)
    private CollegeTier collegeTier;
    
    @Column(nullable = false, precision = 12, scale = 2)
    private BigDecimal coApplicantIncome;
    
    @Column(nullable = false)
    private Integer creditScore;
    
    @Column(nullable = false, precision = 12, scale = 2)
    private BigDecimal existingLiabilities;
    
    @Column(precision = 12, scale = 2)
    private BigDecimal eligibleAmount;
    
    @Column(precision = 5, scale = 2)
    private BigDecimal eligibilityScore;
    
    @Column(precision = 5, scale = 2)
    private BigDecimal debtToIncomeRatio;
    
    @Enumerated(EnumType.STRING)
    private RiskLevel riskLevel;
    
    @CreationTimestamp
    @Column(updatable = false)
    private LocalDateTime createdAt;
    
    public enum CourseType {
        ENGINEERING("Engineering", 0.85),
        MEDICAL("Medical", 0.90),
        MBA("MBA", 0.80),
        LAW("Law", 0.75),
        ARTS("Arts & Humanities", 0.70),
        SCIENCE("Science", 0.75),
        COMMERCE("Commerce", 0.70),
        OTHERS("Others", 0.65);
        
        private final String displayName;
        private final double riskFactor;
        
        CourseType(String displayName, double riskFactor) {
            this.displayName = displayName;
            this.riskFactor = riskFactor;
        }
        
        public String getDisplayName() {
            return displayName;
        }
        
        public double getRiskFactor() {
            return riskFactor;
        }
    }
    
    public enum CollegeTier {
        TIER_1("Tier 1 - Premier Institutions", 1.0),
        TIER_2("Tier 2 - Good Institutions", 0.85),
        TIER_3("Tier 3 - Standard Institutions", 0.70);
        
        private final String displayName;
        private final double multiplier;
        
        CollegeTier(String displayName, double multiplier) {
            this.displayName = displayName;
            this.multiplier = multiplier;
        }
        
        public String getDisplayName() {
            return displayName;
        }
        
        public double getMultiplier() {
            return multiplier;
        }
    }
    
    public enum RiskLevel {
        LOW("Low Risk", "green"),
        MEDIUM("Medium Risk", "yellow"),
        HIGH("High Risk", "red");
        
        private final String displayName;
        private final String color;
        
        RiskLevel(String displayName, String color) {
            this.displayName = displayName;
            this.color = color;
        }
        
        public String getDisplayName() {
            return displayName;
        }
        
        public String getColor() {
            return color;
        }
    }
}
