#!/usr/bin/env python3
"""
UPI Fraud Detection System - Test Cases
Comprehensive test scenarios for the fraud detection system
"""

import requests
import json
import time
from datetime import datetime
import random

# API Configuration
API_BASE_URL = "http://localhost:8002"
FRONTEND_URL = "http://localhost:3000"

def test_health_check():
    """Test 1: Health Check"""
    print("🔍 Test 1: Health Check")
    try:
        response = requests.get(f"{API_BASE_URL}/health")
        print(f"Status Code: {response.status_code}")
        print(f"Response: {response.json()}")
        assert response.status_code == 200
        print("✅ Health check passed!\n")
    except Exception as e:
        print(f"❌ Health check failed: {e}\n")

def test_model_status():
    """Test 2: Model Status"""
    print("🔍 Test 2: Model Status")
    try:
        response = requests.get(f"{API_BASE_URL}/model/status")
        print(f"Status Code: {response.status_code}")
        print(f"Response: {response.json()}")
        assert response.status_code == 200
        print("✅ Model status check passed!\n")
    except Exception as e:
        print(f"❌ Model status check failed: {e}\n")

def test_legitimate_transaction():
    """Test 3: Legitimate Transaction"""
    print("🔍 Test 3: Legitimate Transaction (Should be Safe)")
    
    legitimate_transaction = {
        "transaction_id": "TXN123456789",
        "upi_id": "user@paytm",
        "amount": 500.00,
        "merchant": "Grocery Store",
        "timestamp": datetime.now().isoformat(),
        "location": "Mumbai",
        "device_id": "DEV001"
    }
    
    try:
        response = requests.post(f"{API_BASE_URL}/predict", json=legitimate_transaction)
        print(f"Status Code: {response.status_code}")
        result = response.json()
        print(f"Response: {json.dumps(result, indent=2)}")
        print(f"Fraud Risk: {'HIGH' if result.get('is_fraud', False) else 'LOW'}")
        print("✅ Legitimate transaction test completed!\n")
    except Exception as e:
        print(f"❌ Legitimate transaction test failed: {e}\n")

def test_suspicious_transaction():
    """Test 4: Suspicious Transaction (High Amount)"""
    print("🔍 Test 4: Suspicious Transaction - High Amount")
    
    suspicious_transaction = {
        "transaction_id": "TXN987654321",
        "upi_id": "suspicious@phonepe",
        "amount": 50000.00,  # Very high amount
        "merchant": "Unknown Merchant",
        "timestamp": datetime.now().isoformat(),
        "location": "Unknown",
        "device_id": "DEV999"
    }
    
    try:
        response = requests.post(f"{API_BASE_URL}/predict", json=suspicious_transaction)
        print(f"Status Code: {response.status_code}")
        result = response.json()
        print(f"Response: {json.dumps(result, indent=2)}")
        print(f"Fraud Risk: {'HIGH' if result.get('is_fraud', False) else 'LOW'}")
        print("✅ Suspicious transaction test completed!\n")
    except Exception as e:
        print(f"❌ Suspicious transaction test failed: {e}\n")

def test_night_transaction():
    """Test 5: Night Time Transaction"""
    print("🔍 Test 5: Night Time Transaction (Suspicious Pattern)")
    
    night_transaction = {
        "transaction_id": "TXN_NIGHT_001",
        "upi_id": "nightuser@gpay",
        "amount": 15000.00,
        "merchant": "ATM Withdrawal",
        "timestamp": "2026-02-02T02:30:00",  # 2:30 AM
        "location": "Remote Area",
        "device_id": "DEV_NIGHT"
    }
    
    try:
        response = requests.post(f"{API_BASE_URL}/predict", json=night_transaction)
        print(f"Status Code: {response.status_code}")
        result = response.json()
        print(f"Response: {json.dumps(result, indent=2)}")
        print(f"Fraud Risk: {'HIGH' if result.get('is_fraud', False) else 'LOW'}")
        print("✅ Night transaction test completed!\n")
    except Exception as e:
        print(f"❌ Night transaction test failed: {e}\n")

def test_rapid_transactions():
    """Test 6: Rapid Transaction Pattern"""
    print("🔍 Test 6: Rapid Transaction Pattern (Multiple quick transactions)")
    
    base_transaction = {
        "upi_id": "rapiduser@paytm",
        "merchant": "Online Store",
        "location": "Delhi",
        "device_id": "DEV_RAPID"
    }
    
    for i in range(3):
        transaction = base_transaction.copy()
        transaction.update({
            "transaction_id": f"TXN_RAPID_{i+1}",
            "amount": random.uniform(1000, 5000),
            "timestamp": datetime.now().isoformat()
        })
        
        try:
            response = requests.post(f"{API_BASE_URL}/predict", json=transaction)
            result = response.json()
            print(f"Transaction {i+1}: {result.get('transaction_id')} - Risk: {'HIGH' if result.get('is_fraud', False) else 'LOW'}")
            time.sleep(1)  # 1 second between transactions
        except Exception as e:
            print(f"❌ Rapid transaction {i+1} failed: {e}")
    
    print("✅ Rapid transactions test completed!\n")

def test_metrics():
    """Test 7: Metrics Endpoint"""
    print("🔍 Test 7: System Metrics")
    try:
        response = requests.get(f"{API_BASE_URL}/metrics")
        print(f"Status Code: {response.status_code}")
        print(f"Metrics: {json.dumps(response.json(), indent=2)}")
        assert response.status_code == 200
        print("✅ Metrics test passed!\n")
    except Exception as e:
        print(f"❌ Metrics test failed: {e}\n")

def test_invalid_data():
    """Test 8: Invalid Data Handling"""
    print("🔍 Test 8: Invalid Data Handling")
    
    invalid_transaction = {
        "transaction_id": "INVALID",
        "amount": -100,  # Negative amount
        "upi_id": "",    # Empty UPI ID
    }
    
    try:
        response = requests.post(f"{API_BASE_URL}/predict", json=invalid_transaction)
        print(f"Status Code: {response.status_code}")
        print(f"Response: {response.json()}")
        print("✅ Invalid data handling test completed!\n")
    except Exception as e:
        print(f"❌ Invalid data test failed: {e}\n")

def run_all_tests():
    """Run all test cases"""
    print("🚀 Starting UPI Fraud Detection System Tests")
    print("=" * 60)
    print(f"API Endpoint: {API_BASE_URL}")
    print(f"Frontend URL: {FRONTEND_URL}")
    print("=" * 60)
    
    # Run all tests
    test_health_check()
    test_model_status()
    test_legitimate_transaction()
    test_suspicious_transaction()
    test_night_transaction()
    test_rapid_transactions()
    test_metrics()
    test_invalid_data()
    
    print("🎉 All tests completed!")
    print("\n📊 Frontend Testing:")
    print(f"   • Open {FRONTEND_URL} in your browser")
    print("   • Test the interactive dashboard")
    print("   • Monitor real-time fraud detection")

if __name__ == "__main__":
    run_all_tests()