// Professional UPI Fraud Detection Dashboard JavaScript

class FraudDetectionDashboard {
    constructor() {
        this.apiBaseUrl = 'http://localhost:8000'; // Will auto-detect available port
        this.charts = {};
        this.isActivityPaused = false;
        this.activityItems = [];
        
        this.init();
    }
    
    async init() {
        try {
            // Find active API port
            await this.findActiveAPI();
            
            // Initialize dashboard
            this.initializeEventListeners();
            this.initializeCharts();
            this.initializeTabs();
            this.setupSettingsListeners();
            this.startRealTimeUpdates();
            this.simulateRealTimeActivity();
            
            console.log('✅ Dashboard initialized successfully');
            this.showToast('Dashboard loaded successfully', 'success');
        } catch (error) {
            console.error('❌ Dashboard initialization failed:', error);
            this.showToast('Failed to initialize dashboard', 'error');
        }
    }
    
    async findActiveAPI() {
        const ports = [8000, 8001, 8002, 8003];
        
        for (const port of ports) {
            try {
                const response = await fetch(`http://localhost:${port}/health`, {
                    method: 'GET',
                    timeout: 2000
                });
                
                if (response.ok) {
                    this.apiBaseUrl = `http://localhost:${port}`;
                    console.log(`✅ Found active API at ${this.apiBaseUrl}`);
                    return;
                }
            } catch (error) {
                continue;
            }
        }
        
        // If no API found, use mock mode
        console.warn('⚠️ No active API found, using mock mode');
        this.useMockMode = true;
    }
    
    initializeEventListeners() {
        // Form submission
        const form = document.getElementById('fraudCheckForm');
        if (form) {
            form.addEventListener('submit', (e) => this.handleFormSubmission(e));
        }
        
        // Activity controls
        const pauseBtn = document.getElementById('pauseActivity');
        const clearBtn = document.getElementById('clearActivity');
        
        if (pauseBtn) {
            pauseBtn.addEventListener('click', () => this.toggleActivityFeed());
        }
        
        if (clearBtn) {
            clearBtn.addEventListener('click', () => this.clearActivityFeed());
        }
        
        // Time range selector
        const timeRangeSelect = document.getElementById('timeRange');
        if (timeRangeSelect) {
            timeRangeSelect.addEventListener('change', (e) => this.updateChartsTimeRange(e.target.value));
        }
        
        // Global search
        const globalSearch = document.getElementById('globalSearch');
        if (globalSearch) {
            globalSearch.addEventListener('input', (e) => this.handleGlobalSearch(e.target.value));
        }
    }
    
    initializeCharts() {
        this.initFraudTrendsChart();
        this.initRiskDistributionChart();
        this.initVolumeChart();
    }
    
    initFraudTrendsChart() {
        const ctx = document.getElementById('fraudTrendsChart');
        if (!ctx) return;
        
        this.charts.fraudTrends = new Chart(ctx, {
            type: 'line',
            data: {
                labels: this.generateTimeLabels(24),
                datasets: [
                    {
                        label: 'Safe Transactions',
                        data: this.generateMockData(24, 800, 1200),
                        borderColor: '#10b981',
                        backgroundColor: 'rgba(16, 185, 129, 0.1)',
                        tension: 0.4,
                        fill: true,
                        pointBackgroundColor: '#10b981',
                        pointBorderColor: '#ffffff',
                        pointBorderWidth: 2,
                        pointRadius: 4,
                        pointHoverRadius: 6
                    },
                    {
                        label: 'Fraud Detected',
                        data: this.generateMockData(24, 10, 50),
                        borderColor: '#ef4444',
                        backgroundColor: 'rgba(239, 68, 68, 0.1)',
                        tension: 0.4,
                        fill: true,
                        pointBackgroundColor: '#ef4444',
                        pointBorderColor: '#ffffff',
                        pointBorderWidth: 2,
                        pointRadius: 4,
                        pointHoverRadius: 6
                    },
                    {
                        label: 'Suspicious Activity',
                        data: this.generateMockData(24, 20, 80),
                        borderColor: '#f59e0b',
                        backgroundColor: 'rgba(245, 158, 11, 0.1)',
                        tension: 0.4,
                        fill: true,
                        pointBackgroundColor: '#f59e0b',
                        pointBorderColor: '#ffffff',
                        pointBorderWidth: 2,
                        pointRadius: 4,
                        pointHoverRadius: 6
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                interaction: {
                    intersect: false,
                    mode: 'index'
                },
                plugins: {
                    legend: {
                        position: 'top',
                        align: 'end',
                        labels: {
                            boxWidth: 12,
                            boxHeight: 12,
                            borderRadius: 6,
                            usePointStyle: true,
                            font: {
                                size: 12,
                                weight: '500'
                            }
                        }
                    },
                    tooltip: {
                        mode: 'index',
                        intersect: false,
                        backgroundColor: 'rgba(255, 255, 255, 0.95)',
                        titleColor: '#374151',
                        bodyColor: '#374151',
                        borderColor: '#e5e7eb',
                        borderWidth: 1,
                        cornerRadius: 8,
                        displayColors: true,
                        callbacks: {
                            title: function(context) {
                                return `Time: ${context[0].label}`;
                            },
                            label: function(context) {
                                return `${context.dataset.label}: ${context.parsed.y} transactions`;
                            }
                        }
                    }
                },
                scales: {
                    x: {
                        display: true,
                        title: {
                            display: true,
                            text: 'Time (Hours)',
                            font: {
                                size: 12,
                                weight: '600'
                            }
                        },
                        grid: {
                            color: 'rgba(0, 0, 0, 0.05)'
                        }
                    },
                    y: {
                        display: true,
                        title: {
                            display: true,
                            text: 'Number of Transactions',
                            font: {
                                size: 12,
                                weight: '600'
                            }
                        },
                        grid: {
                            color: 'rgba(0, 0, 0, 0.05)'
                        },
                        beginAtZero: true
                    }
                },
                elements: {
                    line: {
                        borderWidth: 3
                    }
                }
            }
        });
    }
    
    initRiskDistributionChart() {
        const ctx = document.getElementById('riskDistributionChart');
        if (!ctx) return;
        
        this.charts.riskDistribution = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: ['Low Risk', 'Medium Risk', 'High Risk', 'Blocked'],
                datasets: [{
                    data: [87.2, 8.5, 3.1, 1.2],
                    backgroundColor: [
                        '#10b981',
                        '#f59e0b',
                        '#ef4444',
                        '#6b7280'
                    ],
                    borderWidth: 0
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                return context.label + ': ' + context.parsed + '%';
                            }
                        }
                    }
                }
            }
        });
    }
    
    initVolumeChart() {
        const ctx = document.getElementById('volumeChart');
        if (!ctx) return;
        
        this.charts.volume = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                datasets: [{
                    label: 'Transaction Volume',
                    data: [12847, 13952, 14123, 13456, 15678, 11234, 9876],
                    backgroundColor: 'rgba(99, 102, 241, 0.8)',
                    borderColor: '#6366f1',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'Transactions'
                        }
                    }
                }
            }
        });
    }
    
    generateTimeLabels(hours) {
        const labels = [];
        const now = new Date();
        
        for (let i = hours - 1; i >= 0; i--) {
            const time = new Date(now.getTime() - (i * 60 * 60 * 1000));
            labels.push(time.toLocaleTimeString('en-US', { 
                hour: '2-digit', 
                minute: '2-digit',
                hour12: false 
            }));
        }
        
        return labels;
    }
    
    generateMockData(length, min, max) {
        return Array.from({ length }, () => 
            Math.floor(Math.random() * (max - min + 1)) + min
        );
    }
    
    async handleFormSubmission(e) {
        e.preventDefault();
        
        const formData = new FormData(e.target);
        const transactionData = {
            transaction_id: formData.get('transactionId') || document.getElementById('transactionId').value,
            amount: parseFloat(formData.get('amount') || document.getElementById('amount').value),
            merchant: formData.get('merchant') || document.getElementById('merchant').value,
            location: formData.get('location') || document.getElementById('location').value,
            timestamp: new Date().toISOString(),
            upi_id: 'user@example',
            device_id: 'WEB_DASHBOARD'
        };
        
        // Show loading state
        this.setFormLoading(true);
        
        try {
            let result;
            
            if (this.useMockMode) {
                result = this.generateMockAnalysisResult(transactionData);
            } else {
                const response = await fetch(`${this.apiBaseUrl}/predict`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(transactionData)
                });
                
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                
                result = await response.json();
            }
            
            // Display results
            this.displayAnalysisResult(result, transactionData);
            this.addActivityItem(result, transactionData);
            
            // Update stats
            this.updateStats(result);
            
            this.showToast('Transaction analyzed successfully', 'success');
            
        } catch (error) {
            console.error('Analysis failed:', error);
            this.showToast('Analysis failed. Please try again.', 'error');
            
            // Show mock result on error
            const mockResult = this.generateMockAnalysisResult(transactionData);
            this.displayAnalysisResult(mockResult, transactionData);
            
        } finally {
            this.setFormLoading(false);
        }
    }
    
    generateMockAnalysisResult(transactionData) {
        const amount = transactionData.amount;
        let riskScore, isHigh, decision, factors;
        
        // Risk calculation based on amount and other factors
        if (amount > 25000) {
            riskScore = 75 + Math.random() * 20; // 75-95%
            isHigh = true;
            decision = 'BLOCK';
            factors = {
                'Amount': '90.0%',
                'Location': '65.0%',
                'Merchant': '70.0%',
                'Time': '45.0%'
            };
        } else if (amount > 10000) {
            riskScore = 40 + Math.random() * 30; // 40-70%
            isHigh = Math.random() > 0.5;
            decision = isHigh ? 'REVIEW' : 'ALLOW';
            factors = {
                'Amount': '60.0%',
                'Location': '45.0%',
                'Merchant': '35.0%',
                'Time': '25.0%'
            };
        } else {
            riskScore = Math.random() * 25; // 0-25%
            isHigh = false;
            decision = 'ALLOW';
            factors = {
                'Amount': '15.0%',
                'Location': '20.0%',
                'Merchant': '10.0%',
                'Time': '5.0%'
            };
        }
        
        return {
            transaction_id: transactionData.transaction_id,
            is_fraud: isHigh,
            risk_score: riskScore,
            decision: decision,
            confidence: 85 + Math.random() * 10,
            processing_time_ms: 15 + Math.random() * 10,
            risk_factors: factors,
            recommendation: isHigh 
                ? 'High risk transaction detected. Recommend blocking or manual review.' 
                : 'Transaction appears safe. Approved for processing.'
        };
    }
    
    displayAnalysisResult(result, transactionData) {
        const resultContainer = document.getElementById('analysisResult');
        const resultContent = document.getElementById('resultContent');
        const resultTimestamp = document.getElementById('resultTimestamp');
        
        if (!resultContainer || !resultContent) return;
        
        // Update timestamp
        if (resultTimestamp) {
            resultTimestamp.textContent = new Date().toLocaleString();
        }
        
        // Determine risk level
        const riskLevel = result.risk_score > 70 ? 'high' : 
                         result.risk_score > 30 ? 'medium' : 'low';
        
        const riskLevelText = riskLevel.charAt(0).toUpperCase() + riskLevel.slice(1) + ' Risk';
        
        resultContent.innerHTML = `
            <div class="risk-result">
                <div class="risk-header">
                    <h2 class="risk-level ${riskLevel}">${riskLevelText}</h2>
                    <div class="risk-score">${result.risk_score.toFixed(1)}%</div>
                </div>
                
                <div class="risk-details">
                    <div class="risk-detail">
                        <span>Decision:</span>
                        <span class="decision ${result.decision.toLowerCase()}">${result.decision}</span>
                    </div>
                    <div class="risk-detail">
                        <span>Confidence:</span>
                        <span>${result.confidence?.toFixed(1) || '88.0'}%</span>
                    </div>
                    <div class="risk-detail">
                        <span>Processing Time:</span>
                        <span>${result.processing_time_ms?.toFixed(0) || '15'}ms</span>
                    </div>
                </div>
                
                <div class="risk-factors">
                    <h4>Risk Factors</h4>
                    <div class="factor-list">
                        ${Object.entries(result.risk_factors || {}).map(([factor, score]) => `
                            <div class="factor-item">
                                <span class="factor-name">${factor}</span>
                                <span class="factor-score ${this.getFactorClass(parseFloat(score))}">${score}</span>
                            </div>
                        `).join('')}
                    </div>
                </div>
                
                <div class="recommendation">
                    <h4>Recommendation</h4>
                    <p>${result.recommendation || 'Transaction processed successfully.'}</p>
                </div>
            </div>
        `;
        
        resultContainer.style.display = 'block';
    }
    
    getFactorClass(score) {
        if (score > 70) return 'high';
        if (score > 40) return 'medium';
        return 'low';
    }
    
    setFormLoading(isLoading) {
        const button = document.querySelector('.analyze-btn');
        const loader = document.querySelector('.btn-loader');
        const buttonText = button.querySelector('i:first-child');
        
        if (isLoading) {
            button.disabled = true;
            buttonText.style.display = 'none';
            loader.style.display = 'inline';
        } else {
            button.disabled = false;
            buttonText.style.display = 'inline';
            loader.style.display = 'none';
        }
    }
    
    addActivityItem(result, transactionData) {
        if (this.isActivityPaused) return;
        
        const feed = document.getElementById('activityFeed');
        if (!feed) return;
        
        const isBlocked = result.is_fraud || result.risk_score > 70;
        const time = new Date().toLocaleTimeString();
        
        const activityItem = document.createElement('div');
        activityItem.className = 'activity-item';
        activityItem.innerHTML = `
            <div class="activity-icon ${isBlocked ? 'blocked' : 'safe'}">
                <i class="fas ${isBlocked ? 'fa-times' : 'fa-check'}"></i>
            </div>
            <div class="activity-content">
                <div class="activity-title">
                    ${transactionData.transaction_id} - ₹${transactionData.amount.toLocaleString()}
                </div>
                <div class="activity-time">${time} • ${transactionData.merchant}</div>
            </div>
        `;
        
        feed.insertBefore(activityItem, feed.firstChild);
        
        // Keep only last 20 items
        while (feed.children.length > 20) {
            feed.removeChild(feed.lastChild);
        }
    }
    
    simulateRealTimeActivity() {
        // Initial activity items
        this.addInitialActivity();
        
        setInterval(() => {
            if (this.isActivityPaused) return;
            
            const merchants = ['Amazon', 'Flipkart', 'Swiggy', 'Zomato', 'BigBasket', 'Grocery Store', 'Coffee Shop', 'Paytm', 'PhonePe', 'Google Pay'];
            const locations = ['Mumbai', 'Delhi', 'Bangalore', 'Chennai', 'Hyderabad', 'Pune', 'Kolkata'];
            
            const merchant = merchants[Math.floor(Math.random() * merchants.length)];
            const location = locations[Math.floor(Math.random() * locations.length)];
            const amount = Math.floor(Math.random() * 5000) + 100;
            const isRisky = Math.random() < 0.08; // 8% fraud rate
            
            const mockTransaction = {
                transaction_id: `TXN${Date.now().toString().slice(-6)}`,
                amount: amount,
                merchant: merchant,
                location: location
            };
            
            const mockResult = {
                is_fraud: isRisky,
                risk_score: isRisky ? Math.random() * 40 + 60 : Math.random() * 30
            };
            
            this.addActivityItem(mockResult, mockTransaction);
        }, 3000); // Every 3 seconds
    }
    
    addInitialActivity() {
        const initialItems = [
            { transaction_id: 'TXN001', amount: 1500, merchant: 'Amazon', is_fraud: false, risk_score: 15 },
            { transaction_id: 'TXN002', amount: 45000, merchant: 'Unknown', is_fraud: true, risk_score: 85 },
            { transaction_id: 'TXN003', amount: 850, merchant: 'Grocery Store', is_fraud: false, risk_score: 12 },
            { transaction_id: 'TXN004', amount: 2500, merchant: 'Swiggy', is_fraud: false, risk_score: 18 },
            { transaction_id: 'TXN005', amount: 35000, merchant: 'Cash Withdrawal', is_fraud: true, risk_score: 78 }
        ];
        
        initialItems.forEach((item, index) => {
            setTimeout(() => {
                this.addActivityItem(
                    { is_fraud: item.is_fraud, risk_score: item.risk_score },
                    { transaction_id: item.transaction_id, amount: item.amount, merchant: item.merchant }
                );
            }, index * 500);
        });
    }
    
    toggleActivityFeed() {
        this.isActivityPaused = !this.isActivityPaused;
        const button = document.getElementById('pauseActivity');
        const icon = button.querySelector('i');
        
        if (this.isActivityPaused) {
            icon.className = 'fas fa-play';
            button.title = 'Resume Activity';
        } else {
            icon.className = 'fas fa-pause';
            button.title = 'Pause Activity';
        }
    }
    
    clearActivityFeed() {
        const feed = document.getElementById('activityFeed');
        if (feed) {
            feed.innerHTML = '';
        }
        this.showToast('Activity feed cleared', 'info');
    }
    
    updateStats(result) {
        // Update header stats
        if (result.is_fraud) {
            const fraudCount = document.getElementById('fraudBlocked');
            if (fraudCount) {
                const current = parseInt(fraudCount.textContent) || 0;
                fraudCount.textContent = (current + 1).toString();
            }
        } else {
            const safeCount = document.getElementById('safeTransactions');
            if (safeCount) {
                const current = parseInt(safeCount.textContent.replace(',', '')) || 0;
                safeCount.textContent = (current + 1).toLocaleString();
            }
        }
    }
    
    updateChartsTimeRange(range) {
        // Update chart data based on time range
        console.log(`Updating charts for time range: ${range}`);
        
        let hours;
        switch (range) {
            case '24h':
                hours = 24;
                break;
            case '7d':
                hours = 24 * 7;
                break;
            case '30d':
                hours = 24 * 30;
                break;
            default:
                hours = 24;
        }
        
        // Update fraud trends chart
        if (this.charts.fraudTrends) {
            this.charts.fraudTrends.data.labels = this.generateTimeLabels(Math.min(hours, 24));
            this.charts.fraudTrends.data.datasets[0].data = this.generateMockData(Math.min(hours, 24), 800, 1200);
            this.charts.fraudTrends.data.datasets[1].data = this.generateMockData(Math.min(hours, 24), 10, 50);
            this.charts.fraudTrends.data.datasets[2].data = this.generateMockData(Math.min(hours, 24), 20, 80);
            this.charts.fraudTrends.update();
        }
    }
    
    handleGlobalSearch(query) {
        if (query.length < 3) return;
        
        console.log(`Searching for: ${query}`);
        // Implement search functionality
        this.showToast(`Searching for "${query}"...`, 'info');
    }
    
    startRealTimeUpdates() {
        // Update stats periodically
        setInterval(() => {
            // Simulate real-time updates
            this.updateRandomStats();
        }, 10000); // Every 10 seconds
    }
    
    updateRandomStats() {
        // Simulate random stat updates
        const elements = [
            { id: 'safeTransactions', min: 1200, max: 1300 },
            { id: 'fraudBlocked', min: 20, max: 30 }
        ];
        
        elements.forEach(({ id, min, max }) => {
            const element = document.getElementById(id);
            if (element && Math.random() < 0.3) { // 30% chance to update
                const current = parseInt(element.textContent.replace(',', '')) || min;
                const increment = Math.floor(Math.random() * 5) + 1;
                element.textContent = (current + increment).toLocaleString();
            }
        });
    }
    
    showToast(message, type = 'info') {
        const container = document.getElementById('toastContainer');
        if (!container) return;
        
        const toast = document.createElement('div');
        toast.className = `toast ${type}`;
        toast.innerHTML = `
            <div style="display: flex; align-items: center; gap: 0.5rem;">
                <i class="fas ${this.getToastIcon(type)}"></i>
                <span>${message}</span>
            </div>
        `;
        
        container.appendChild(toast);
        
        // Remove toast after 4 seconds
        setTimeout(() => {
            if (container.contains(toast)) {
                container.removeChild(toast);
            }
        }, 4000);
    }
    
    getToastIcon(type) {
        switch (type) {
            case 'success': return 'fa-check-circle';
            case 'error': return 'fa-times-circle';
            case 'warning': return 'fa-exclamation-triangle';
            default: return 'fa-info-circle';
        }
    }

    initializeTabs() {
        const tabs = document.querySelectorAll('.tab-btn');
        const contents = document.querySelectorAll('.tab-content');

        tabs.forEach(tab => {
            tab.addEventListener('click', () => {
                const target = tab.getAttribute('data-tab');
                
                // Remove active class from all tabs and contents
                tabs.forEach(t => t.classList.remove('active'));
                contents.forEach(c => c.classList.remove('active'));
                
                // Add active class to clicked tab and corresponding content
                tab.classList.add('active');
                const targetContent = document.getElementById(target);
                if (targetContent) {
                    targetContent.classList.add('active');
                }
            });
        });

        // Activate first tab by default
        if (tabs.length > 0) {
            tabs[0].click();
        }
    }

    setupSettingsListeners() {
        // Handle slider value updates
        const sliders = document.querySelectorAll('.threshold-slider');
        sliders.forEach(slider => {
            slider.addEventListener('input', (e) => {
                const valueDisplay = e.target.parentElement.querySelector('.slider-value');
                if (valueDisplay) {
                    valueDisplay.textContent = e.target.value;
                }
            });
        });

        // Handle save settings button
        const saveBtn = document.querySelector('.btn-settings:not(.reset)');
        if (saveBtn) {
            saveBtn.addEventListener('click', this.saveSettings.bind(this));
        }

        // Handle reset settings button
        const resetBtn = document.querySelector('.btn-settings.reset');
        if (resetBtn) {
            resetBtn.addEventListener('click', this.resetSettings.bind(this));
        }

        // Handle model actions
        const modelActions = document.querySelectorAll('.btn-model');
        modelActions.forEach(btn => {
            btn.addEventListener('click', (e) => {
                const action = e.target.textContent;
                this.handleModelAction(action, e.target);
            });
        });

        // Handle alert actions
        const alertActions = document.querySelectorAll('.btn-alert');
        alertActions.forEach(btn => {
            btn.addEventListener('click', (e) => {
                const action = e.target.textContent;
                this.handleAlertAction(action, e.target);
            });
        });
    }

    saveSettings() {
        const settings = {
            fraud_threshold: document.getElementById('fraud-threshold').value,
            security_level: document.getElementById('security-level').value,
            notifications: document.getElementById('notifications').checked,
            realtime_monitoring: document.getElementById('realtime-monitoring').checked,
            auto_alerts: document.getElementById('auto-alerts').checked,
            model_retraining: document.getElementById('model-retraining').checked,
            log_level: document.getElementById('log-level').value,
            backup_frequency: document.getElementById('backup-frequency').value
        };

        // Show saving status
        const statusDiv = document.querySelector('.processing-status');
        if (statusDiv) {
            statusDiv.innerHTML = '<div class="status-indicator"></div>Saving settings...';
            
            setTimeout(() => {
                statusDiv.innerHTML = '<div class="status-indicator" style="background: var(--success);"></div>Settings saved successfully!';
            }, 1500);
        }

        console.log('Saved settings:', settings);
    }

    resetSettings() {
        // Reset all form elements to defaults
        document.getElementById('fraud-threshold').value = 75;
        document.querySelector('#fraud-threshold').parentElement.querySelector('.slider-value').textContent = '75';
        document.getElementById('security-level').value = 'high';
        document.getElementById('notifications').checked = true;
        document.getElementById('realtime-monitoring').checked = true;
        document.getElementById('auto-alerts').checked = false;
        document.getElementById('model-retraining').checked = true;
        document.getElementById('log-level').value = 'info';
        document.getElementById('backup-frequency').value = 'daily';

        // Show reset status
        const statusDiv = document.querySelector('.processing-status');
        if (statusDiv) {
            statusDiv.innerHTML = '<div class="status-indicator" style="background: var(--warning);"></div>Settings reset to defaults';
        }
    }

    handleModelAction(action, button) {
        const modelCard = button.closest('.model-card');
        const modelName = modelCard.querySelector('h3').textContent;
        
        // Simulate action
        button.disabled = true;
        button.textContent = 'Processing...';
        
        setTimeout(() => {
            button.disabled = false;
            button.textContent = action;
            console.log(`${action} executed for ${modelName}`);
        }, 2000);
    }

    handleAlertAction(action, button) {
        const alertItem = button.closest('.alert-item');
        
        if (action === 'Resolve') {
            alertItem.style.opacity = '0.5';
            button.textContent = 'Resolved';
            button.disabled = true;
        } else if (action === 'View') {
            console.log('Viewing alert details');
        }
    }
}

// Initialize dashboard when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    new FraudDetectionDashboard();
});
