import axios from 'axios';
import {
    authRequest,
    authSuccess,
    authFailed,
    authError,
    userDetailsSuccess,
    onRequest,
    doneSuccess,
    getFailed,
    getError,
    addedSuccess,
    employeesListSuccess,
    employeeDetailsSuccess,
} from './userSlice';

const BASE_URL = process.env.REACT_APP_BASE_URL || 'http://localhost:5000';

export const authHandler = (fields, role, mode) => async (dispatch) => {
    dispatch(authRequest());

    try {
        // debug: print the final URL and payload so frontend console shows what's being requested
        console.log('authHandler: POST', `${BASE_URL}/${role}${mode}`, fields);
        const result = await axios.post(`${BASE_URL}/${role}${mode}`, fields);
        if (result.data.message) {
            dispatch(authFailed(result.data.message));
        } else {
            dispatch(authSuccess(result.data));
        }
    } catch (error) {
        // log full error to browser console for debugging (network issues, CORS, etc.)
        console.error('authHandler error:', error);
        // normalize error message for the app state
        if (error.response && error.response.data && error.response.data.message) {
            dispatch(authError(error.response.data.message));
        } else if (error.message) {
            // axios network errors often include a message
            dispatch(authError(error.message));
        } else {
            dispatch(authError('Network Error'));
        }
    }
};

export const addStuff = (address, fields, token) => async (dispatch) => {
    dispatch(authRequest());

    try {
        const result = await axios.post(`${process.env.REACT_APP_BASE_URL}/${address}`, fields, {
            headers: { Authorization: token }
        })
        if (result.data.message) {
            dispatch(authFailed(result.data.message));
        } else {
            dispatch(addedSuccess());
        }
    } catch (error) {
        dispatch(authError(error));
    }
};

export const getUserDetails = (address, token) => async (dispatch) => {
    dispatch(onRequest());

    try {
        const result = await axios.get(`${process.env.REACT_APP_BASE_URL}/${address}`, {
            headers: { Authorization: token }
        });

        if (result.data.message) {
            dispatch(getFailed(result.data.message));
        } else {
            dispatch(userDetailsSuccess(result.data));
        }
    } catch (error) {
        dispatch(getError(error));
    }
};

export const multipleDeleteHandler = (address, token) => async (dispatch) => {
    dispatch(onRequest());

    try {
        const result = await axios.delete(`${process.env.REACT_APP_BASE_URL}/${address}`, {
            headers: { Authorization: token }
        });

        if (result.data.message) {
            dispatch(getFailed(result.data.message));
        } else {
            dispatch(doneSuccess());
        }
    } catch (error) {
        dispatch(getError(error));
    }
}

export const singleDeleteHandler = (address, id, token) => async (dispatch) => {
    dispatch(onRequest());

    try {
        const result = await axios.delete(`${process.env.REACT_APP_BASE_URL}/${address}/${id}`, {
            headers: { Authorization: token }
        });

        if (result.data.message) {
            dispatch(getFailed(result.data.message));
        } else {
            dispatch(doneSuccess());
        }
    } catch (error) {
        dispatch(getError(error));
    }
}

export const updateUser = (fields, token, address) => async (dispatch) => {
    dispatch(onRequest());

    try {
        const result = await axios.put(`${process.env.REACT_APP_BASE_URL}/${address}`, fields, {
            headers: {
                Authorization: token
            }
        });

        if (result.data.message) {
            dispatch(authFailed(result.data.message));
        } else {
            dispatch(authSuccess(result.data));
        }
    } catch (error) {
        dispatch(getError(error));
    }
}

export const getAllEmployees = (address, token) => async (dispatch) => {
    dispatch(onRequest());

    try {
        const result = await axios.get(`${process.env.REACT_APP_BASE_URL}/${address}`, {
            headers: { Authorization: token }
        });

        if (result.data.message) {
            dispatch(getFailed(result.data.message));
        } else {
            dispatch(employeesListSuccess(result.data));
        }
    } catch (error) {
        dispatch(getError(error));
    }
}

export const getEmployeeDetails = (address, id, token) => async (dispatch) => {
    dispatch(onRequest());

    try {
        const result = await axios.get(`${process.env.REACT_APP_BASE_URL}/${address}/${id}`, {
            headers: { Authorization: token }
        });

        if (result.data.message) {
            dispatch(getFailed(result.data.message));
        } else {
            dispatch(employeeDetailsSuccess(result.data));
        }

    } catch (error) {
        dispatch(getError(error));
    }
};